<?php

/*
 * i-Educar - Sistema de gesto escolar
 *
 * Copyright (C) 2006  Prefeitura Municipal de Itaja
 *                     <ctima@itajai.sc.gov.br>
 *
 * Este programa  software livre; voc pode redistribu-lo e/ou modific-lo
 * sob os termos da Licena Pblica Geral GNU conforme publicada pela Free
 * Software Foundation; tanto a verso 2 da Licena, como (a seu critrio)
 * qualquer verso posterior.
 *
 * Este programa  distribudo na expectativa de que seja til, porm, SEM
 * NENHUMA GARANTIA; nem mesmo a garantia implcita de COMERCIABILIDADE OU
 * ADEQUAO A UMA FINALIDADE ESPECFICA. Consulte a Licena Pblica Geral
 * do GNU para mais detalhes.
 *
 * Voc deve ter recebido uma cpia da Licena Pblica Geral do GNU junto
 * com este programa; se no, escreva para a Free Software Foundation, Inc., no
 * endereo 59 Temple Street, Suite 330, Boston, MA 02111-1307 USA.
 */

/**
 * CoreExt_Config class.
 *
 * Transforma um array comum em um objeto CoreExt_Config, permitindo que esse
 * array seja acessado com o operador ->, assim como variveis de instncia.
 * Dessa forma, um array como:
 *
 * <code>
 * $array = array(
 *   'app' => array(
 *     'database' => array(
 *       'dbname'   => 'ieducar',
 *       'hostname' => 'localhost',
 *       'password' => 'ieducar',
 *       'username' => 'ieducar',
 *     )
 *   ),
 *   'CoreExt' => '1'
 * )
 * </code>
 *
 * Pode ser acessado dessa forma:
 *
 * <code>
 * $config = new CoreExt_Config($array);
 * print $config->app->database->dbname;
 * </code>
 *
 * Essa classe foi fortemente baseada na classe Zend_Config do Zend Framework s
 * que implementa menos funcionalidades.
 *
 * @author      Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @license     http://creativecommons.org/licenses/GPL/2.0/legalcode.pt  CC GNU GPL
 * @package     CoreExt
 * @subpackage  Config
 * @since       Classe disponvel desde a verso 1.1.0
 * @version     $Id$
 */
class CoreExt_Config implements Countable, Iterator
{

  /**
   * Array de sobre sobrecarga
   * @var array
   */
  protected $config;

  /**
   * Array com mensagens de erro causadas por funes PHP.
   * @var array
   */
  protected $errors = array();

  /**
   * ndice interno do array para a implementao da interface Iterator.
   * @var int
   */
  private $_index = 0;

  /**
   * Quantidade de items do array de sobrecarga $config para a implementao da interface Countable.
   * @var int
   */
  private $_count = 0;

  /**
   * Construtor da classe.
   *
   * @param  $array  Array associativo com as diretivas de configurao.
   */
  public function __construct($array)
  {
    foreach ($array as $key => $val) {
      if (is_array($val)) {
        $this->config[$key] = new self($val);
      }
      else {
        $this->config[$key] = $val;
      }
    }

    $this->_count = count($this->config);
  }

  /**
   * Retorna o valor do array de sobrecarga $config.
   *
   * Este mtodo deve ser usado toda vez que a varivel de configurao puder
   * ser sobrescrita por um storage de configurao externa ao cdigo, como o
   * arquivo ini. Um exemplo seria para a criao de um arquivo on the fly no
   * filesystem. No cdigo pode ser assumido que o local padro ser
   * intranet/tmp mas, se esse valor puder ser sobrescrito pelo ini, esse mtodo
   * dever ser utilizado:
   * <code>
   * $dir = $config->get($config->app->filesystem->tmp_dir, 'intranet/tmp');
   * </code>
   *
   * Se a varivel de configurao no for sobrescrita por um arquivo ini ou
   * array de configurao, o valor padro (segundo parmetro) ser utilizado.
   *
   * @param   mixed  $value1  Valor retornado pelo array de configurao sobrecarregado
   * @param   mixed  $value2  Valor padro caso no exista uma configurao sobrecarregada
   * @return  mixed
   */
  public function get($value1, $value2 = NULL)
  {
    if (NULL != $value1) {
      return $value1;
    }

    if (NULL == $value2) {
      throw new Exception('O segundo parmetro deve conter algum valor no nulo.');
    }

    return $value2;
  }

  /**
   * Retorna o valor armazenado pelo ndice no array de sobrecarga $config.
   *
   * @param   $key    ndice (nome) da varivel criada por sobrecarga
   * @param   $value  Valor padro caso o ndice no exista
   * @return  mixed   O valor armazenado em
   */
  private function getFrom($key, $value = NULL)
  {
    if (array_key_exists($key, $this->config)) {
      $value = $this->config[$key];
    }

    return $value;
  }

  /**
   * Implementao do mtodo mgico __get().
   *
   * @param $key
   * @return unknown_type
   */
  public function __get($key) {
    return $this->getFrom($key);
  }

  /**
   * Retorna o contedo do array de sobrecarga em um array associativo simples.
   * @return  array
   */
  public function toArray()
  {
    $array = array();
    foreach ($this->config as $key => $value) {
      $array[$key] = $value;
    }
    return $array;
  }

  /**
   * Implementao do mtodo count() da interface Countable.
   */
  public function count() {
    return $this->_count;
  }

  /**
   * Implementao do mtodo next() da interface Iterator.
   */
  public function next() {
    next($this->config);
    ++$this->_index;
  }

  /**
   * Implementao do mtodo next() da interface Iterator.
   */
  public function rewind() {
    reset($this->config);
    $this->_index = 0;
  }

  /**
   * Implementao do mtodo current() da interface Iterator.
   */
  public function current() {
    return current($this->config);
  }

  /**
   * Implementao do mtodo key() da interface Iterator.
   */
  public function key() {
    return key($this->config);
  }

  /**
   * Implementao do mtodo valid() da interface Iterator.
   */
  public function valid() {
    return $this->_index < $this->_count && $this->_index > -1;
  }

  /**
   * Merge recursivo mantendo chaves distintas.
   *
   * Realiza um merge recursivo entre dois arrays.  semelhante a funo PHP
   * {@link http://php.net/array_merge_recursive array_merge_recursive} exceto
   * pelo fato de que esta mantm apenas um valor de uma chave do array ao invs
   * de criar mltiplos valores para a mesma chave como na funo original.
   *
   * @author  Daniel Smedegaard Buus <daniel@danielsmedegaardbuus.dk>
   * @link    http://www.php.net/manual/pt_BR/function.array-merge-recursive.php#89684  Cdigo fonte original
   * @param   array  $arr1
   * @param   array  $arr2
   * @return  array
   */
  protected function &arrayMergeRecursiveDistinct(&$arr1, &$arr2)
  {
    $merged = $arr1;

    if (is_array($arr2)) {
      foreach ($arr2 as $key => $val) {
        if (is_array($arr2[$key])) {
          $merged[$key] = is_array($merged[$key]) ?
            $this->arrayMergeRecursiveDistinct($merged[$key], $arr2[$key]) : $arr2[$key];
        }
        else {
          $merged[$key] = $val;
        }
      }
    }

    return $merged;
  }

  /**
   * Mtodo callback para a funo set_error_handler().
   *
   * Handler para os erros internos da classe. Dessa forma,  possvel usar
   * os blocos try/catch para lanar excees.
   *
   * @see  http://php.net/set_error_handler
   * @param  $errno
   * @param  $errstr
   */
  protected function configErrorHandler($errno, $errstr) {
    $this->errors[] = array($errno => $errstr);
  }

}