<?php

/*
 * i-Educar - Sistema de gesto escolar
 *
 * Copyright (C) 2006  Prefeitura Municipal de Itaja
 *                     <ctima@itajai.sc.gov.br>
 *
 * Este programa  software livre; voc pode redistribu-lo e/ou modific-lo
 * sob os termos da Licena Pblica Geral GNU conforme publicada pela Free
 * Software Foundation; tanto a verso 2 da Licena, como (a seu critrio)
 * qualquer verso posterior.
 *
 * Este programa  distribudo na expectativa de que seja til, porm, SEM
 * NENHUMA GARANTIA; nem mesmo a garantia implcita de COMERCIABILIDADE OU
 * ADEQUAO A UMA FINALIDADE ESPECFICA. Consulte a Licena Pblica Geral
 * do GNU para mais detalhes.
 *
 * Voc deve ter recebido uma cpia da Licena Pblica Geral do GNU junto
 * com este programa; se no, escreva para a Free Software Foundation, Inc., no
 * endereo 59 Temple Street, Suite 330, Boston, MA 02111-1307 USA.
 */

/**
 * CoreExt_Config_Ini class.
 *
 * Essa classe torna possvel o uso de um arquivo .ini como meio de configurao
 * da aplicao. O parsing do arquivo  feito atravs da funo PHP nativa
 * parse_ini_file. Possibilita o uso de herana simples e separao de
 * namespaces no arquivo ini, tornando simples a criao de diferentes espaos
 * de configurao para o uso em ambientes diversos como produo,
 * desenvolvimento, testes e outros.
 *
 * Para o uso dessa classe,  necessrio que o arquivo ini tenha no mnimo uma
 * seo de configurao. A seo padro a ser usada  a production mas isso
 * no impede que voc a nomeie como desejar.
 *
 * Essa classe foi fortemente baseada na classe Zend_Config_Ini do Zend
 * Framework.
 *
 * @author      Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @license     http://creativecommons.org/licenses/GPL/2.0/legalcode.pt  CC GNU GPL
 * @package     CoreExt
 * @subpackage  Config
 * @see         lib/CoreExt/Config.class.php
 * @since       Classe disponvel desde a verso 1.1.0
 * @version     $Id$
 */
class CoreExt_Config_Ini extends CoreExt_Config
{

  /**
   * Caractere de herana das sees do arquivo ini.
   */
  const COREEXT_CONFIG_INI_INHERITANCE_SEP = ':';

  /**
   * Caractere de namespace das diretivas do arquivo ini.
   */
  const COREEXT_CONFIG_INI_NAMESPACE_SEP   = '.';

  /**
   * Array contendo as diretivas de configurao separadas por namespace.
   * @var array
   */
  protected $iniArr = array();

  /**
   * Construtor.
   *
   * @param  $filename  Caminho para o arquivo ini
   * @param  $section   Seo desejada para o carregamento das configuraes
   */
  public function __construct($filename, $section = 'production')
  {
    require_once 'CoreExt/Config.class.php';

    $this->iniArr = $this->loadFile($filename);
    parent::__construct($this->iniArr[$section]);
  }

  /**
   * Carrega as configuraes para o ambiente desejado (seo do arquivo ini)
   * @param  string  $section
   */
  public function changeEnviroment($section = 'production') {
    $this->changeSection($section);
  }

  /**
   * Verifica se possui a seo desejada.
   * @param  string  $section
   */
  function hasEnviromentSection($section) {
    return is_array($this->iniArr[$section]);
  }

  /**
   * Carrega as configurao da seo desejada.
   * @param  string  $section
   */
  protected function changeSection($section = 'production') {
    parent::__construct($this->iniArr[$section]);
  }

  /**
   * Parsing do arquivo ini.
   *
   * Realiza o parsing do arquivo ini, separando as sees, criando as relaes
   * de herana e separando cada diretiva do arquivo em um array
   * multidimensional.
   *
   * @link    http://php.net/parse_ini_file  Documentao da funo parse_ini_file
   * @param   string  $filename
   * @throws  Exception
   * @return  array
   */
  private function loadFile($filename)
  {
    $iniArr = array();

    // Faz o parsing separando as sees (parmetro TRUE)
    // Supresso simples dificulta os unit test. Altera o error handler para
    // que use um mtodo da classe CoreExt_Config.
    set_error_handler(array($this, 'configErrorHandler'), E_ALL);
    $config = parse_ini_file($filename, TRUE);
    restore_error_handler();

    /*
     * No PHP 5.2.7 o array vem FALSE quando existe um erro de sintaxe. Antes
     * o retorno vinha como array vazio.
     * @link  http://php.net/parse_ini_file#function.parse-ini-file.changelog  Changelog da funo parse_ini_file
     */
    if (count($this->errors) > 0) {
      throw new Exception('Arquivo ini com problemas de sintaxe. Verifique a sintaxe arquivo \''. $filename .'\'.');
    }

    foreach ($config as $section => $data) {
      $index = $section;

      if (FALSE !== strpos($section, self::COREEXT_CONFIG_INI_INHERITANCE_SEP)) {
        $sections = explode(self::COREEXT_CONFIG_INI_INHERITANCE_SEP, $section);
        // Apenas uma herana por seo  permitida
        if (count($sections) > 2) {
          throw new Exception('No  possvel herdar mais que uma seo.');
        }

        // Armazena seo atual e seo de herana
        $section = trim($sections[0]);
        $extends = trim($sections[1]);
      }

      // Processa as diretivas da seo atual para separarar os namespaces
      $iniArr[$section] = $this->processSection($config[$index]);

      /*
       * Verifica se a seo atual herda de alguma outra seo. Se a seo de
       * herana no existir, lana uma exceo.
       */
      if (isset($extends)) {
        if (!array_key_exists($extends, $iniArr)) {
          $message = sprintf('No foi possvel estender %s, seo %s no existe',
            $section, $extends);
          throw new Exception($message);
        }

        // Mescla recursivamente os dois arrays. Os valores definidos na seo
        // atual no so sobrescritos
        $iniArr[$section] = $this->arrayMergeRecursiveDistinct($iniArr[$extends], $iniArr[$section]);
        unset($extends);
      }
    }

    return $iniArr;
  }

  /**
   * Processa uma seo de um array de arquivo ini.
   *
   * Processa a seo, inclusive as diretivas da seo, separando-as em
   * um array em namespace. Dessa forma, uma diretiva que era, por exemplo,
   * app.database.dbname = ieducardb ir se tornar:
   * <code>
   * app => array(database => array(dbname => ieducardb))
   * </code>
   *
   * Diretivas no mesmo namespace viram novas chaves no mesmo array:
   * app.database.hostname => localhost
   * <code>
   * app => array(
   *   database => array(
   *     dbname   => ieducardb
   *     hostname => localhost
   *   )
   * )
   * </code>
   *
   * @param   array  $data  Array contendo as diretivas de uma seo do arquivo ini
   * @return  array
   */
  private function processSection(array $data)
  {
    $entries = $data;
    $config  = array();

    foreach ($entries as $key => $value) {
      if (FALSE !== strpos($key, self::COREEXT_CONFIG_INI_NAMESPACE_SEP)) {
        $keys = explode(self::COREEXT_CONFIG_INI_NAMESPACE_SEP, $key);
      }
      else {
        $keys = (array) $key;
      }

      $config = $this->processDirectives($value, $keys, $config);
    }

    return $config;
  }

  /**
   * Cria recursivamente um array aninhado (namespaces) usando os ndices
   * progressivamente.
   *
   * Exemplo:
   * <code>
   * $value  = ieducardb
   * $keys   = array('app', 'database', 'dbname');
   *
   * $config['app'] => array(
   *   'database' => array(
   *     'dbname' => 'ieducardb'
   *   )
   * );
   * </code>
   *
   * @param   mixed  $value   O valor da diretiva parseada por parse_ini_file
   * @param   array  $keys    O array contendo as chaves das diretivas (0 => app, 1 => database, 2 => dbname)
   * @param   array  $config  O array continer com as chaves em suas respectivas dimenses
   * @return  array
   */
  private function processDirectives($value, $keys, $config = array())
  {
    $key = array_shift($keys);

    if (count($keys) > 0) {
      if (!array_key_exists($key, $config)) {
        $config[$key] = array();
      }

      $config[$key] = $this->processDirectives($value, $keys, $config[$key]);
    }
    else {
      $config[$key] = $value;
    }

    return $config;
  }

}