<?php

/**
 * i-Educar - Sistema de gesto escolar
 *
 * Copyright (C) 2006  Prefeitura Municipal de Itaja
 *                     <ctima@itajai.sc.gov.br>
 *
 * Este programa  software livre; voc pode redistribu-lo e/ou modific-lo
 * sob os termos da Licena Pblica Geral GNU conforme publicada pela Free
 * Software Foundation; tanto a verso 2 da Licena, como (a seu critrio)
 * qualquer verso posterior.
 *
 * Este programa  distribudo na expectativa de que seja til, porm, SEM
 * NENHUMA GARANTIA; nem mesmo a garantia implcita de COMERCIABILIDADE OU
 * ADEQUAO A UMA FINALIDADE ESPECFICA. Consulte a Licena Pblica Geral
 * do GNU para mais detalhes.
 *
 * Voc deve ter recebido uma cpia da Licena Pblica Geral do GNU junto
 * com este programa; se no, escreva para a Free Software Foundation, Inc., no
 * endereo 59 Temple Street, Suite 330, Boston, MA 02111-1307 USA.
 *
 * @author    Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @category  i-Educar
 * @license   @@license@@
 * @package   CoreExt_Controller
 * @since     Arquivo disponvel desde a verso 1.1.0
 * @version   $Id$
 */

require_once 'CoreExt/Controller/Abstract.php';

/**
 * CoreExt_Controller_Front class.
 *
 * Essa  uma implementao simples do design pattern {@link http://martinfowler.com/eaaCatalog/frontController.html front controller},
 * que tem como objetivo manusear e encaminhar a requisio para uma classe
 * que se responsabilize pelo processamento do recurso solicitado.
 *
 * Apesar de ser um front controller, o encaminhamento para uma classe
 * {@link http://en.wikipedia.org/wiki/Command_pattern command} no est
 * implementado.
 *
 * Entretanto, est disponvel o encaminhamento para uma classe que implemente
 * o pattern {@link http://martinfowler.com/eaaCatalog/pageController.html page controller},
 * ou seja, qualquer classe que implemente a interface
 * CoreExt_Controller_Page_Interface.
 *
 * O processo de encaminhamento (dispatching),  definido por uma classe
 * {@link http://en.wikipedia.org/wiki/Strategy_pattern strategy}.
 *
 * Algumas opes afetam o comportamento dessa classe. As opes disponveis
 * para configurar uma instncia da classe so:
 * - basepath: diretrio em que os implementadores de command e page controller
 *   sero procurados
 * - controller_dir: determina o nome do diretrio em que os controllers devero
 *   estar salvos
 * - controller_type: tipo de controller a ser instanciado. Uma instncia de
 *   CoreExt_Controller_Front pode usar apenas um tipo por processo de
 *   dispatch() e o valor dessa opo determina qual strategy de dispatch ser
 *   utilizada (CoreExt_Controller_Strategy).
 *
 * Por padro, os valores de controller_dir e controller_type so definidos para
 * 'Views' e 2, respectivamente. Isso significa que a estratgia de page
 * controller ser utilizada durante a chamada ao mtodo dispatch().
 *
 * @author    Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @category  i-Educar
 * @license   @@license@@
 * @package   CoreExt_Controller
 * @since     Classe disponvel desde a verso 1.1.0
 * @version   @@package_version@@
 */
class CoreExt_Controller_Front extends CoreExt_Controller_Abstract
{
  /**
   * Opes para definio de qual tipo de controller utilizar durante a
   * execuo de dispatch().
   * @var int
   */
  const CONTROLLER_FRONT = 1;
  const CONTROLLER_PAGE  = 2;

  /**
   * A instncia singleton de CoreExt_Controller_Interface.
   * @var CoreExt_Controller_Interface|NULL
   */
  protected static $_instance = NULL;

  /**
   * Opes de configurao geral da classe.
   * @var array
   */
  protected $_options = array(
    'basepath'        => NULL,
    'controller_type' => self::CONTROLLER_PAGE,
    'controller_dir'  => 'Views'
  );

  /**
   * Contm os valores padro da configurao.
   * @var array
   */
  protected $_defaultOptions = array();

  /**
   * Uma instncia de CoreExt_View_Abstract
   * @var CoreExt_View_Abstract
   */
  protected $_view = NULL;

  /**
   * Construtor singleton.
   */
  protected function __construct()
  {
    $this->_defaultOptions = $this->getOptions();
  }

  /**
   * Retorna a instncia singleton.
   * @return CoreExt_Controller_Front
   */
  public static function getInstance()
  {
    if (is_null(self::$_instance)) {
      self::$_instance = new self();
    }
    return self::$_instance;
  }

  /**
   * Recupera os valores de configurao original da instncia.
   * @return CoreExt_Configurable Prov interface fluda
   */
  public function resetOptions()
  {
    $this->setOptions($this->_defaultOptions);
    return $this;
  }

  /**
   * Encaminha a execuo para o objeto CoreExt_Dispatcher_Interface apropriado.
   * @return CoreExt_Controller_Interface Prov interface fluda
   * @see CoreExt_Controller_Interface#dispatch()
   */
  public function dispatch()
  {
    $this->_getControllerStrategy()->dispatch();
    return $this;
  }

  /**
   * Retorna o contedo gerado pelo controller.
   * @return string
   */
  public function getViewContents()
  {
    return $this->getView()->getContents();
  }

  /**
   * Setter.
   * @param CoreExt_View_Abstract $view
   * @return CoreExt_Controller_Interface Prov interface fluda
   */
  public function setView(CoreExt_View_Abstract $view)
  {
    $this->_view = $view;
    return $this;
  }

  /**
   * Getter para uma instncia de CoreExt_View_Abstract.
   *
   * Instncia via lazy initialization uma instncia de CoreExt_View caso
   * nenhuma seja explicitamente atribuda a instncia atual.
   *
   * @return CoreExt_View_Abstract
   */
  public function getView()
  {
    if (is_null($this->_view)) {
      require_once 'CoreExt/View.php';
      $this->setView(new CoreExt_View());
    }
    return $this->_view;
  }

  /**
   * Getter para uma instncia de CoreExt_Controller_Dispatcher_Interface.
   *
   * Instncia via lazy initialization uma instncia de
   * CoreExt_Controller_Dispatcher caso nenhuma seja explicitamente
   * atribuda a instncia atual.
   *
   * @return CoreExt_Controller_Dispatcher_Interface
   */
  public function getDispatcher()
  {
    if (is_null($this->_dispatcher)) {
      $this->setDispatcher($this->_getControllerStrategy());
    }
    return $this->_dispatcher;
  }

  /**
   * Getter para a estratgia de controller, definida em runtime.
   * @return CoreExt_Controller_Strategy
   */
  protected function _getControllerStrategy()
  {
    switch($this->getOption('controller_type')) {
      case 1:
        require_once 'CoreExt/Controller/Dispatcher/Strategy/FrontStrategy.php';
        $strategy = 'CoreExt_Controller_Dispatcher_Strategy_FrontStrategy';
        break;
      case 2:
        require_once 'CoreExt/Controller/Dispatcher/Strategy/PageStrategy.php';
        $strategy = 'CoreExt_Controller_Dispatcher_Strategy_PageStrategy';
        break;
    }
    return new $strategy($this);
  }
}