<?php

/**
 * i-Educar - Sistema de gesto escolar
 *
 * Copyright (C) 2006  Prefeitura Municipal de Itaja
 *                     <ctima@itajai.sc.gov.br>
 *
 * Este programa  software livre; voc pode redistribu-lo e/ou modific-lo
 * sob os termos da Licena Pblica Geral GNU conforme publicada pela Free
 * Software Foundation; tanto a verso 2 da Licena, como (a seu critrio)
 * qualquer verso posterior.
 *
 * Este programa  distribudo na expectativa de que seja til, porm, SEM
 * NENHUMA GARANTIA; nem mesmo a garantia implcita de COMERCIABILIDADE OU
 * ADEQUAO A UMA FINALIDADE ESPECFICA. Consulte a Licena Pblica Geral
 * do GNU para mais detalhes.
 *
 * Voc deve ter recebido uma cpia da Licena Pblica Geral do GNU junto
 * com este programa; se no, escreva para a Free Software Foundation, Inc., no
 * endereo 59 Temple Street, Suite 330, Boston, MA 02111-1307 USA.
 *
 * @author    Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @category  i-Educar
 * @license   @@license@@
 * @package   CoreExt_Session
 * @since     Arquivo disponvel desde a verso 1.1.0
 * @version   $Id$
 */

require_once 'CoreExt/Session/Storage/Interface.php';

/**
 * CoreExt_Session_Storage_Abstract abstract class.
 *
 * Implementa operaes bsicas para facilitar a implementao de
 * CoreExt_Session_Storage_Interface.
 *
 * Opes disponveis:
 * - session_name: o nome da session, o padro  o valor definido no php.ini
 * - session_auto_start: se a session deve ser iniciada na instanciao da
 *   classe. Padro  TRUE
 * - session_auto_shutdown: se um mtodo de shutdown deve ser chamado no
 *   encerramento da execuo do script PHP. Padro  TRUE.
 *
 * @author    Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @category  i-Educar
 * @license   @@license@@
 * @package   CoreExt_Session
 * @since     Classe disponvel desde a verso 1.1.0
 * @version   @@package_version@@
 */
abstract class CoreExt_Session_Storage_Abstract
  implements CoreExt_Session_Storage_Interface, Countable
{
  /**
   * Flag para definir se a session foi iniciada ou no, til para impedir que
   * funes que enviem headers sejam chamadas novamente (session_start, p.ex.)
   * @var bool
   */
  protected static $_sessionStarted = FALSE;

  /**
   * Id da session atual.
   * @var string
   */
  protected static $_sessionId = NULL;

  /**
   * Opes de configurao geral da classe.
   * @var array
   */
  protected $_options = array(
    'session_name'          => NULL,
    'session_auto_start'    => TRUE,
    'session_auto_shutdown' => TRUE
  );

  /**
   * Construtor.
   * @param array $options Array de opes de configurao.
   */
  public function __construct(array $options = array())
  {
    $this->_init($options);

    if (TRUE == $this->getOption('session_auto_shutdown')) {
      register_shutdown_function(array($this, 'shutdown'));
    }
  }

  /**
   * Mtodo de inicializao do storage. As subclasses devem sobrescrever
   * este mtodo para alterar o comportamento do mecanismo de session do PHP.
   *
   * @return CoreExt_Session_Storage_Abstract Prov interfae fluda
   */
  protected function _init(array $options = array())
  {
    $this->setOptions($options);
  }

  /**
   * @see CoreExt_Configurable#setOptions($options)
   */
  public function setOptions(array $options = array())
  {
    $this->_options = array_merge($this->getOptions(), $options);
    return $this;
  }

  /**
   * @see CoreExt_Configurable#getOptions()
   */
  public function getOptions()
  {
    return $this->_options;
  }

  /**
   * Verifica se uma opo est setada.
   *
   * @param string $key
   * @return bool
   */
  protected function _hasOption($key)
  {
    return array_key_exists($key, $this->_options);
  }

  /**
   * Retorna um valor de opo de configurao ou NULL caso a opo no esteja
   * setada.
   *
   * @param string $key
   * @return mixed|NULL
   */
  public function getOption($key)
  {
    return $this->_hasOption($key) ? $this->_options[$key] : NULL;
  }

  /**
   * Getter.
   * @return string
   */
  public static function getSessionId()
  {
    return self::$_sessionId;
  }

  /**
   * Getter.
   * @return bool
   */
  public static function isStarted()
  {
    return self::$_sessionStarted;
  }

  /**
   * Getter.
   *
   * Deve ser implementado pelas subclasses para retornar o array de dados
   * persistidos na session, permitindo que clientes iterem livremente pelos
   * dados.
   *
   * @return array
   */
  public abstract function getSessionData();
}