<?php

/**
 * i-Educar - Sistema de gesto escolar
 *
 * Copyright (C) 2006  Prefeitura Municipal de Itaja
 *                     <ctima@itajai.sc.gov.br>
 *
 * Este programa  software livre; voc pode redistribu-lo e/ou modific-lo
 * sob os termos da Licena Pblica Geral GNU conforme publicada pela Free
 * Software Foundation; tanto a verso 2 da Licena, como (a seu critrio)
 * qualquer verso posterior.
 *
 * Este programa  distribudo na expectativa de que seja til, porm, SEM
 * NENHUMA GARANTIA; nem mesmo a garantia implcita de COMERCIABILIDADE OU
 * ADEQUAO A UMA FINALIDADE ESPECFICA. Consulte a Licena Pblica Geral
 * do GNU para mais detalhes.
 *
 * Voc deve ter recebido uma cpia da Licena Pblica Geral do GNU junto
 * com este programa; se no, escreva para a Free Software Foundation, Inc., no
 * endereo 59 Temple Street, Suite 330, Boston, MA 02111-1307 USA.
 *
 * @author    Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @category  i-Educar
 * @license   @@license@@
 * @package   CoreExt_Validate
 * @since     Arquivo disponvel desde a verso 1.1.0
 * @version   $Id$
 */

require_once 'CoreExt/Validate/Interface.php';

/**
 * CoreExt_Validate_Abstract abstract class.
 *
 * @author    Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @category  i-Educar
 * @license   @@license@@
 * @package   CoreExt_Validate
 * @since     Classe disponvel desde a verso 1.1.0
 * @version   @@package_version@@
 */
abstract class CoreExt_Validate_Abstract implements CoreExt_Validate_Interface
{
  /**
   * Opes de configurao geral da classe.
   * @var array
   */
  protected $_options = array(
    'required' => TRUE,
    'trim' => TRUE,
  );

  /**
   * Valor no sanitizado que foi informado ao validador.
   * @var mixed
   */
  protected $_value = NULL;

  /**
   * Valor sanitizado.
   * @var mixed
   */
  protected $_sanitized = NULL;

  /**
   * Mensagem padro para erros de valor obrigatrio.
   * @var string
   */
  protected $_requiredMessage = 'Obrigatrio.';

  /**
   * Mensagem padro para erros de invalidez.
   * @var string
   */
  protected $_invalidMessage = 'Invlido.';

  /**
   * Construtor.
   *
   * Pode receber array com opes de configurao da classe.
   *
   * @param array $options
   */
  public function __construct(array $options = array())
  {
    $this->_options = array_merge($this->getOptions(), $this->_getDefaultOptions());
    $this->setOptions($options);
  }

  /**
   * Configura as opes do validador.
   *
   * Mtodo de checagem de opes inspirado na tcnica empregada no
   * {@link http://www.symfony-project.org symfony framework}.
   *
   * @param  array $options
   * @throws InvalidArgumentException Lana exceo no verificada caso alguma
   *   opo passada ao mtodo no exista na definio da classe
   */
  public function setOptions(array $options = array())
  {
    $defaultOptions = array_keys($this->getOptions());
    $passedOptions  = array_keys($options);

    if (0 < count(array_diff($passedOptions, $defaultOptions))) {
      throw new InvalidArgumentException(
        sprintf('A classe %s no suporta as opes: %s.', get_class($this), implode(', ', $passedOptions))
      );
    }

    $this->_options = array_merge($this->getOptions(), $options);
  }

  /**
   * @see CoreExt_Validate_Interface#getOptions()
   */
  public function getOptions()
  {
    return $this->_options;
  }

  /**
   * Verifica se uma opo est setada.
   *
   * @param string $key
   * @return bool
   */
  protected function _hasOption($key)
  {
    return
      isset($this->_options[$key]) &&
      !$this->_isEmpty($this->_options[$key]);
  }

  /**
   * Retorna um valor de opo de configurao ou NULL caso a opo no esteja
   * setada.
   *
   * @param string $key
   * @return mixed|NULL
   */
  public function getOption($key)
  {
    return $this->_hasOption($key) ? $this->_options[$key] : NULL;
  }

  /**
   * Permite que uma classe que estenda CoreExt_Validate_Abstract a definir
   * valores de opes pr-definidos adequados ao caso especfico.
   *
   * @return array
   */
  protected abstract function _getDefaultOptions();

  /**
   * @see CoreExt_Validate_Interface#isValid($value)
   */
  public function isValid($value)
  {
    $this->_value = $value;
    $value = $this->_sanitize($value);

    if (TRUE == $this->getOption('trim')) {
      $value = trim($value);
    }

    $this->_sanitized = $value;

    if (TRUE == $this->getOption('required') && $this->_isEmpty($value)) {
      throw new Exception($this->_requiredMessage);
    }

    return $this->_validate($value);
  }

  /**
   * Toda classe que estende CoreExt_Validate_Abstract deve implementar esse
   * mtodo com a lgica de validao adequada.
   *
   * @param  string $value
   * @return bool
   */
  protected abstract function _validate($value);

  /**
   * Realiza uma sanitizao
   * @param mixed $value
   * @return mixed
   */
  protected function _sanitize($value)
  {
    return $value;
  }

  /**
   * Verifica se um dado valor est vazio.
   *
   * Como vazio, entende-se string vazia (''), array sem itens (array()), o
   * valor NULL e zero (0) numrico.
   *
   * @param  mixed $value
   * @return bool
   */
  protected function _isEmpty($value)
  {
    return in_array(
      $value, array('', array(), NULL), TRUE
    );
  }

  /**
   * Retorna uma mensagem de erro configurada em $_options.
   *
   * A mensagem de erro pode ser uma string ou um array. Se for uma string,
   * ocorrer a substituio dos placeholders. Se for um array, dever ser
   * especificado duas mensagens de erro, uma para a forma singular e outra
   * para o plural. O placeholder @value ser verificado para definir se a
   * mensagem deve ser formatada no plural ou no singular.
   *
   * Exemplo de array de mensagem de erro que usa variante de nmero:
   *
   * <code>
   * <?php
   * $message = array(
   *   array(
   *     'singular' => '@value problema encontrado.'
   *     'plural'   => '@value problemas encontrados.'
   *   )
   * );
   *
   * // Iria imprimir:
   * // singular (@value = 1): 1 problema encontrado
   * // plural (@value = 4): 4 problemas encontrados
   * </code>
   *
   * @param  array|string  $key      O identificador da mensagem no array $_options
   * @param  array         $options  Array associativo para substituio de valores
   * @return string
   * @todo   Implementar substituio com formatao padro, semelhante ao
   *   a funo Drupal {@link http://api.drupal.org/t t()}.
   * @todo   Implementar formatao singular/plural em uma classe diferente,
   *         como mtodo pblico, permitindo realizar o teste.
   */
  protected function _getErrorMessage($key, array $options = array())
  {
    $message = $this->getOption($key);

    if (is_array($message)) {
      // Verifica o tipo de @value para determinar a quantidade de $count
      if (is_array($options['@value'])) {
        $count = count($options['@value']);
      }
      elseif (is_numeric($options['@value'])) {
        $count = count($options['@value']);
      }
      else {
        $count = 1;
      }

      if (1 < $count) {
        $message = $message['plural'];
        $options['@value'] = implode(', ', $options['@value']);
      }
      else {
        $message = $message['singular'];
        $options['@value'] = array_shift($options['@value']);
      }
    }

    return strtr($message, $options);
  }

  /**
   * @see CoreExt_Validate_Interface#getValue()
   */
  public function getValue()
  {
    return $this->_value;
  }

  /**
   * @see CoreExt_Validate_Interface#getSanitizedValue()
   */
  public function getSanitizedValue()
  {
    return $this->_sanitized;
  }
}