<?php

/*
 * i-Educar - Sistema de gesto escolar
 *
 * Copyright (C) 2006  Prefeitura Municipal de Itaja
 *                     <ctima@itajai.sc.gov.br>
 *
 * Este programa  software livre; voc pode redistribu-lo e/ou modific-lo
 * sob os termos da Licena Pblica Geral GNU conforme publicada pela Free
 * Software Foundation; tanto a verso 2 da Licena, como (a seu critrio)
 * qualquer verso posterior.
 *
 * Este programa  distribudo na expectativa de que seja til, porm, SEM
 * NENHUMA GARANTIA; nem mesmo a garantia implcita de COMERCIABILIDADE OU
 * ADEQUAO A UMA FINALIDADE ESPECFICA. Consulte a Licena Pblica Geral
 * do GNU para mais detalhes.
 *
 * Voc deve ter recebido uma cpia da Licena Pblica Geral do GNU junto
 * com este programa; se no, escreva para a Free Software Foundation, Inc., no
 * endereo 59 Temple Street, Suite 330, Boston, MA 02111-1307 USA.
 */

/**
 * FileStream class.
 *
 * Classe para stream de arquivos implementando checagens de segurana como
 * diretrios permitidos e tipo de arquivo.
 *
 * @author   Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @license  http://creativecommons.org/licenses/GPL/2.0/legalcode.pt  CC GNU GPL
 * @package  Core
 * @since    Classe disponvel desde a verso 1.1.0
 * @version  $Id$
 */
class FileStream
{

  /**
   * Instncia da classe Mimetype
   * @var Mimetype
   */
  protected $Mimetype    = NULL;

  /**
   * Caminho do arquivo para stream
   * @var string
   */
  protected $filepath    = NULL;

  /**
   * Array de diretrios permitidos para stream de arquivos.
   * @var array
   */
  protected $allowedDirs = array();


  /**
   * Construtor.
   *
   * @param  Mimetype  $mimetype     Objeto Mimetype
   * @param  array     $allowedDirs  Diretrios permitidos para stream
   */
  public function __construct(Mimetype $mimetype, array $allowedDirs = array()) {
    $this->Mimetype = $mimetype;
    $this->setAllowedDirectories((array) $allowedDirs);
  }

  /**
   * Configura o nome do arquivo, verificando se o mesmo encontra-se em um
   * diretrio de acesso permitido e se  legvel.
   *
   * @param   string  $filepath  O caminho completo ou relativo do arquivo
   * @throws  Exception
   */
  public function setFilepath($filepath) {
    $this->isReadable($filepath);
    $this->filepath = $filepath;
  }

  /**
   * Configura os diretrios permitidos para stream de arquivos.
   *
   * @param   array  $v
   */
  protected function setAllowedDirectories($v) {
    $this->allowedDirs = $v;
  }

  /**
   * Verifica se o arquivo  legvel e se est em um diretrio permitido
   * para stream de arquivos.
   *
   * @param   string  $filepath  O caminho completo ou relativo ao arquivo
   * @throws  Exception
   */
  protected function isReadable($filepath)
  {
    $fileinfo = pathinfo($filepath);

    if (! $this->isDirectoryAllowed($fileinfo['dirname'])) {
      throw new Exception('Acesso ao diretrio negado.');
    }

    if (! is_readable($filepath)) {
      throw new Exception('Arquivo no existe.');
    }
  }

  /**
   * Verifica se o diretrio est na lista de diretrios permitidos para
   * stream de arquivos.
   *
   * @param   string  $directory
   * @return  bool    Retorna TRUE se o diretrio  permitido
   */
  public function isDirectoryAllowed($directory)
  {
    if (FALSE === array_search($directory, $this->allowedDirs)) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Faz o stream do arquivo.
   *
   * @throws  Exception
   */
  public function streamFile()
  {
    $mimetype = $this->Mimetype->getType($this->filepath);

    if (FALSE === $mimetype) {
      throw new Exception('Extenso no suportada.');
    }

    // Headers para stream de arquivo
    header('Content-Description: File Transfer');
    header('Content-Type: ' . $mimetype);
    header('Content-Disposition: attachment; filename='.basename($this->filepath));
    header('Content-Transfer-Encoding: binary');
    header('Expires: 0');
    header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
    header('Pragma: public');
    header('Content-Length: ' . filesize($this->filepath));
    ob_clean();
    flush();

    // L o arquivo para stream buffer
    readfile($this->filepath);
  }

}