<?php

/**
 * i-Educar - Sistema de gesto escolar
 *
 * Copyright (C) 2006  Prefeitura Municipal de Itaja
 *                     <ctima@itajai.sc.gov.br>
 *
 * Este programa  software livre; voc pode redistribu-lo e/ou modific-lo
 * sob os termos da Licena Pblica Geral GNU conforme publicada pela Free
 * Software Foundation; tanto a verso 2 da Licena, como (a seu critrio)
 * qualquer verso posterior.
 *
 * Este programa  distribudo na expectativa de que seja til, porm, SEM
 * NENHUMA GARANTIA; nem mesmo a garantia implcita de COMERCIABILIDADE OU
 * ADEQUAO A UMA FINALIDADE ESPECFICA. Consulte a Licena Pblica Geral
 * do GNU para mais detalhes.
 *
 * Voc deve ter recebido uma cpia da Licena Pblica Geral do GNU junto
 * com este programa; se no, escreva para a Free Software Foundation, Inc., no
 * endereo 59 Temple Street, Suite 330, Boston, MA 02111-1307 USA.
 *
 * @author    Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @category  i-Educar
 * @license   @@license@@
 * @package   Tests
 * @since     Arquivo disponvel desde a verso 1.1.0
 * @version   $Id$
 */

require_once 'PHPUnit/Framework.php';
require_once 'PHPUnit/Runner/IncludePathTestCollector.php';

/**
 * TestCollector abstract class.
 *
 * Classe abstrata que prov um ponto de extenso para classes de definio de
 * sute de testes do PHPUnit (veja {@link Orga})
 *
 * Ao estender corretamente essa classe, todos as classes de teste do diretrio
 * da classe de definio de sute de testes sero adicionados  sute,
 * tornando desnecessrio a nessidade de usar os construtores de linguagem
 * require e include para incluir esses arquivos.
 *
 * Para estender essa classe, basta informar o caminho para o arquivo da classe
 * de definio da sute na varivel protegida $_file, exemplo:
 *
 * <code>
 * class App_Model_AllTests extends TestCollector
 * {
 *   protected $_file = __FILE__;
 * }
 * </code>
 *
 * Isso  o suficiente para conseguir coletar todos os arquivos do diretrio.
 * Para criar uma sute de testes com todas as classes de teste do diretrio,
 * basta criar uma instncia da classe e chamar o mtodo addDirectoryTests():
 *
 * <code>
 * public static function suite()
 * {
 *   $instance = new self();
 *   return $instance->createTestSuite('App_Model: testes unitrios')
 *                   ->addDirectoryTests();
 * }
 * </code>
 *
 * Se a varivel de instncia $_name estiver sobrescrita, ela ser utilizada
 * por padro caso o mtodo createTestSuite() seja chamado sem o parmetro nome.
 * Dessa forma, basta chamar o mtodo addDirectoryTests():
 *
 * <code>
 * protected $_name = 'App_model: testes unitrios';
 *
 * public static function suite()
 * {
 *   $instance = new self();
 *   return $instance->addDirectoryTests();
 * }
 * </code>
 *
 * @author    Eriksen Costa Paixo <eriksen.paixao_bs@cobra.com.br>
 * @category  i-Educar
 * @license   @@license@@
 * @package   Tests
 * @since     Classe disponvel desde a verso 1.1.0
 * @version   @@package_version@@
 */
abstract class TestCollector
{
  /**
   * Caminho completo do arquivo da classe que estende TestCollector.
   * @var string
   */
  protected $_file = NULL;

  /**
   * Diretrio onde residem os arquivos com as classes de teste.
   * @var array
   */
  protected $_directory = array();

  /**
   * Nome da sute de testes.
   * @var string
   */
  protected $_name  = '';

  /**
   * Uma instncia de PHPUnit_Framework_TestSuite.
   * @var PHPUnit_Framework_TestSuite
   */
  protected $_suite = NULL;

  /**
   * Construtor.
   * @return TestCollector
   */
  public function __construct()
  {
    $this->_defineCurrentDirectory();
  }

  /**
   * Cria um objeto PHPUnit_Framework_TestSuite com o nome passado como
   * argumento ou usando a varivel de instncia $_name.
   *
   * @param   string  $name  O nome para a sute de testes
   * @return  TestCollector  Interface fluda
   * @throws  InvalidArgumentException
   */
  public function createTestSuite($name = '')
  {
    if ((trim((string) $name)) == '' && $this->_name == '') {
      throw new InvalidArgumentException('A classe concreta deve sobrescrever a '
                . 'varivel "$_name" ou passar um nome vlido ao chamar o mtodo'
                . 'createTestSuite().');
    }
    if (trim((string) $name) != '') {
      $name = $this->_name;
    }

    $this->_suite = new PHPUnit_Framework_TestSuite($name);
    return $this;
  }

  /**
   * Adiciona os testes do diretrio da classe de definio de sute.
   *
   * @param   PHPUnit_Framework_TestSuite  $suite
   * @return  PHPUnit_Framework_TestSuite
   */
  public function addDirectoryTests(PHPUnit_Framework_TestSuite $suite = NULL)
  {
    // Se no existir um objeto PHPUnit_Framework_TestSuite, cria um com o nome
    // do arquivo da classe de definio da sute
    if ($this->_suite == NULL && $suite == NULL) {
      $this->createTestSuite();
    }
    if ($suite == NULL) {
      $suite = $this->_suite;
    }

    $suite->addTestFiles($this->_collectTests());
    return $suite;
  }

  /**
   * Retorna um PHPUnit_Util_FilterIterator que contm as regras de incluso
   * de testes do diretrio definido por $_fir.
   *
   * @return PHPUnit_Util_FilterIterator
   */
  protected function _collectTests()
  {
    $testCollector = new PHPUnit_Runner_IncludePathTestCollector($this->_directory);
    return $testCollector->collectTests();
  }

  /**
   * Define o diretrio atual da classe que estende TestCollector. O diretrio 
   * definido pela varivel de instncia $_file.
   *
   * @throws  Exception  Lana exceo
   * @todo    Refatorar o cdigo para utilizar {@link http://php.net/lsb Late static binding}
   *          quando a verso do PHP for a 5.3.
   */
  protected function _defineCurrentDirectory()
  {
    if ($this->_file === NULL) {
      throw new Exception('A classe concreta deve sobrescrever a varivel "$_file".');
    }
    $directory = $this->_getDirectoryPath($this->_file);
    if (!array_search($directory, $this->_directory)) {
      $this->_directory[] = $directory;
    }
  }

  /**
   * Pega o caminho do diretrio que ser varrido para a incluso de testes.
   * @param  string $path
   * @return string
   */
  protected function _getDirectoryPath($path)
  {
    $directory = realpath(dirname($path));
    if (!is_dir($directory)) {
      throw new Exception('The path "'. $directory .'" is not a valid directory');
    }
    return $directory;
  }

  public function addDirectory($directory)
  {
    $this->_directory[] = $this->_getDirectoryPath($directory);
  }
}