package br.com.centralit.citcorpore.negocio;

import java.sql.Timestamp;
import java.util.List;

import br.com.centralit.citcorpore.bean.CalculoJornadaDTO;
import br.com.centralit.citcorpore.bean.ControleSlaDTO;
import br.com.centralit.citcorpore.bean.SolicitacaoServicoDTO;
import br.com.centralit.citcorpore.integracao.ControleSlaDao;
import br.com.centralit.citcorpore.util.Util;
import br.com.citframework.excecao.PersistenceException;
import br.com.citframework.excecao.ServiceException;
import br.com.citframework.integracao.CrudDAO;
import br.com.citframework.integracao.TransactionControler;
import br.com.citframework.service.CrudServiceImpl;
import br.com.citframework.service.ServiceLocator;
import br.com.citframework.util.UtilDatas;

@SuppressWarnings("unchecked")
public class ControleSlaServiceEjb extends CrudServiceImpl implements ControleSlaService {

	private ControleSlaDao controleSlaDao;
	private SolicitacaoServicoService solicitacaoServicoService;

	@Override
	protected CrudDAO<ControleSlaDTO> getDao() {
		if (controleSlaDao == null) {
			controleSlaDao = new ControleSlaDao();
		}
		return controleSlaDao;
	}
	
	private ControleSlaDao getControleSlaDao() {
		if (controleSlaDao == null) {
			controleSlaDao = new ControleSlaDao();
		}
		return controleSlaDao;
	}

	private SolicitacaoServicoService getSolicitacaoServicoService() throws ServiceException {
		if (solicitacaoServicoService == null) {
			solicitacaoServicoService = (SolicitacaoServicoService) ServiceLocator.getInstance().getService(SolicitacaoServicoService.class, null);
		}
		return solicitacaoServicoService;
	}

	private ControleSlaDTO criarNovoControleSla(SolicitacaoServicoDTO solicitacaoServico, final TransactionControler tc) {
		ControleSlaDTO controleSla = new ControleSlaDTO();
		
		controleSla.setIdSolicitacaoServico(solicitacaoServico.getIdSolicitacaoServico());
		if (solicitacaoServico.getDataHoraInicioSLA()==null){
			controleSla.setDataInicial(solicitacaoServico.getDataHoraSolicitacao());
		} else {
		controleSla.setDataInicial(solicitacaoServico.getDataHoraInicioSLA());
		}
		controleSla.setIdStatusInicial(solicitacaoServico.getIdStatus());
		controleSla.setIdTarefa(solicitacaoServico.getIdTarefa());
		controleSla.setIdUsuario(solicitacaoServico.getUsuarioDto().getIdUsuario());
		controleSla.setTempoTotal(0d);
		
		try {
			getDao().setTransactionControler(tc);
			controleSla = getDao().create(controleSla);
		} catch (PersistenceException e) {
			e.printStackTrace();
		}
		
		return controleSla;
	}
	
	@Override
	public ControleSlaDTO recuperarUltimoControleSla(SolicitacaoServicoDTO solicitacaoServico) {
		return getControleSlaDao().consultarUltimoRegistroDeSlaDaSolicitacaoDeServico(solicitacaoServico.getIdSolicitacaoServico(), solicitacaoServico.getIdTarefa());
		}
		
	/**
	 * Tempo util de atendimento dentro da jornada de trabalho, num determinado perodo.
	 * 
	 * @author gilberto.nery
	 * @since 2016.11.17
	 * @param idCalendario
	 * @param dataInicial
	 * @param dataFinal
	 * @param tc
	 * @return
	 */
	private double calcularTempoTotal(Integer idCalendario, Timestamp dataInicial, Timestamp dataFinal, final TransactionControler tc) {

		try {
			CalculoJornadaDTO calculoDto = new CalculoJornadaDTO(idCalendario, dataInicial, dataFinal);
			calculoDto = new CalendarioServiceEjb().calculaTempoDecorridoPelaJornadaDeTrabalho(calculoDto, tc);

			return calculoDto.getTempoDecorridoTotalEmSegundos();
		}catch(Exception e){
			e.printStackTrace();
		}
		return 0;
	}
	
	@Override
	public ControleSlaDTO fechaUltimoControleSla(SolicitacaoServicoDTO solicitacaoServico, final TransactionControler tc) {
		Integer idStatusControleSla = solicitacaoServico.getIdStatusControleSla();
		ControleSlaDTO controleSla = recuperarUltimoControleSla(solicitacaoServico);
		
		if(controleSla == null || controleSla.getIdControleSla() == null){
			controleSla = criarNovoControleSla(solicitacaoServico, tc);
		}
		
		fechaControleSla(solicitacaoServico, tc, idStatusControleSla, controleSla);

		return controleSla;
		}
		
	@Override
	public void registraControleSla(SolicitacaoServicoDTO solicitacaoServico, final TransactionControler tc) {
		ControleSlaDTO controleSla = new ControleSlaDTO();
		
		controleSla.setIdSolicitacaoServico(solicitacaoServico.getIdSolicitacaoServico());
		controleSla.setDataInicial(solicitacaoServico.getDataFinalUltimoControleSla() != null ? solicitacaoServico.getDataFinalUltimoControleSla() : UtilDatas.getDataHoraAtual());
		controleSla.setIdStatusInicial(solicitacaoServico.getIdStatusControleSla());
		controleSla.setIdTarefa(solicitacaoServico.getIdTarefa());
		controleSla.setIdUsuario(solicitacaoServico.getUsuarioDto().getIdUsuario());
		controleSla.setTempoTotal(0d);
		
		try {
			getDao().setTransactionControler(tc);
			getDao().create(controleSla);
		} catch (PersistenceException e) {
			e.printStackTrace();
		}
	}
	
	@Override
	public List<ControleSlaDTO> fechaRegistrosAbertosControleSla(SolicitacaoServicoDTO solicitacaoServico, TransactionControler tc) {
		List<ControleSlaDTO> registrosControleSla = getControleSlaDao().consultaRegistrosAbertosControleSla(solicitacaoServico.getIdSolicitacaoServico());
		Integer idStatusControleSla = solicitacaoServico.getIdStatusControleSla();
	
		if (registrosControleSla != null && !registrosControleSla.isEmpty()) {
			for (ControleSlaDTO controleSla : registrosControleSla) {
				fechaControleSla(solicitacaoServico, tc, idStatusControleSla, controleSla);
		}
	}
	
		return registrosControleSla;
	}

	private void fechaControleSla(SolicitacaoServicoDTO solicitacaoServico, final TransactionControler tc, Integer idStatusControleSla, ControleSlaDTO controleSla) {
		controleSla.setDataFinal(UtilDatas.getDataHoraAtual());
		controleSla.setTempoTotal(calcularTempoTotal(solicitacaoServico.getIdCalendario(), controleSla.getDataInicial(), controleSla.getDataFinal(), tc));
		controleSla.setIdStatusFinal(idStatusControleSla);

		try{
			getDao().setTransactionControler(tc);
			getDao().update(controleSla);
		} catch (PersistenceException e) {
			e.printStackTrace();
		}
	}
	
	@Override
	public void atualizarDadosDeTempoDeAtendimentoDaSolicitacaoDeServico(SolicitacaoServicoDTO solicitacaoServico, final TransactionControler tc){
		final Timestamp dataHoraInicioSLA = (solicitacaoServico.getDataHoraInicioSLA()!=null?solicitacaoServico.getDataHoraInicioSLA():solicitacaoServico.getDataHoraSolicitacao());
		Double prazo = getControleSlaDao().somarRegistrosDeSlaDaSolicitacaoDeServico(solicitacaoServico.getIdSolicitacaoServico(),dataHoraInicioSLA);
			try {
				getSolicitacaoServicoService().atualizarTemposDeAtendimento(solicitacaoServico.getIdSolicitacaoServico(), prazo, tc);
			} catch (ServiceException e) {
				e.printStackTrace();
			}
	}

	@Override
	public void calcularTempoDeAtendimento(SolicitacaoServicoDTO solicitacaoServico) {
		final Timestamp dataHoraInicioSLA = (solicitacaoServico.getDataHoraInicioSLA()!=null?solicitacaoServico.getDataHoraInicioSLA():solicitacaoServico.getDataHoraSolicitacao());
		Double prazo = getControleSlaDao().somarRegistrosDeSlaDaSolicitacaoDeServico(solicitacaoServico.getIdSolicitacaoServico(),dataHoraInicioSLA);
		solicitacaoServico.setTempoAtendimentoHH(new Integer(Util.getHoraHHMMSS(prazo)));
		solicitacaoServico.setTempoAtendimentoMM(new Integer(Util.getMinutoHHMMSS(prazo)));
	}
	
	/**
	 * Atualiza a data hora limite da solicitacao e os prazoHH e prazoMM e caso necessario a datahoralimite
	 * 
	 * @param idSolicitacaoServico
	 */
	public void atualizarDataHoraLimiteETempoDeAtrasoDaSolicitacao(Integer idSolicitacaoServico){
		try {
			SolicitacaoServicoDTO solicitacao = getSolicitacaoServicoService().restoreAll(idSolicitacaoServico);
			if(solicitacao == null || solicitacao.getIdSolicitacaoServico() == null){
				return;
			}
			
			List<ControleSlaDTO> listaControleSla = (List<ControleSlaDTO>) getControleSlaDao().findByIdSolicitacaoServicoAndContabilizaSla(idSolicitacaoServico);
			Timestamp dataHoraLimiteSla = recuperarDataHoraLimiteSla(listaControleSla, solicitacao);
			double tempoDeAtraso;

			if(dataHoraLimiteSla != null){
				tempoDeAtraso = UtilDatas.getIntervaloEntreDatasEmSegundos(dataHoraLimiteSla, solicitacao.getDataHoraFim());
				if(tempoDeAtraso <= 0){
					tempoDeAtraso = 0;
				}
				getSolicitacaoServicoService().atualizarDataHoraLimiteETempoDeAtrasoDaSolicitacaoServico(idSolicitacaoServico, tempoDeAtraso, dataHoraLimiteSla, getDao().getTransactionControler());
			} else {
				tempoDeAtraso = 0;
				getSolicitacaoServicoService().atualizarTempoDeAtrasoDaSolicitacaoServico(idSolicitacaoServico, tempoDeAtraso, getDao().getTransactionControler());
			}
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	
	/**
	 * Retorna a data de inicio de atraso de uma solicitacao pelas atividades de controle de SLA
	 * 
	 * @param listaControleSla
	 * @param solicitacao
	 * @return
	 */
	public Timestamp recuperarDataHoraLimiteSla(List<ControleSlaDTO> listaControleSla, SolicitacaoServicoDTO solicitacao){
		Timestamp dataHoraFimControleSla = null;
		Integer tempoDoSLA = Util.converteHHMMEmSegundos(solicitacao.getPrazoHH(), solicitacao.getPrazoMM());
		Integer tempoCalculado;
		Integer tempoTotalDaAtividadeDeControleSLA;
		
		try {
			for(ControleSlaDTO controle : listaControleSla){
				
				dataHoraFimControleSla = controle.getDataFinal();
				
				tempoTotalDaAtividadeDeControleSLA = controle.getTempoTotal().intValue();
				tempoCalculado = tempoDoSLA - tempoTotalDaAtividadeDeControleSLA;
				
				if(tempoCalculado <= 0){
					return new CalendarioServiceEjb().retornaDataHoraFinal(solicitacao.getIdCalendario(), controle.getDataInicial(), new Integer(Util.getHoraHHMMSS(tempoDoSLA)), new Integer(Util.getMinutoHHMMSS(tempoDoSLA)),getDao().getTransactionControler());
				} else {
					tempoDoSLA -= tempoTotalDaAtividadeDeControleSLA;
				}
			}
			return new CalendarioServiceEjb().retornaDataHoraFinal(solicitacao.getIdCalendario(), dataHoraFimControleSla, new Integer(Util.getHoraHHMMSS(tempoDoSLA)), new Integer(Util.getMinutoHHMMSS(tempoDoSLA)),getDao().getTransactionControler());
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}
	
	/**
	 * Verifica se a solicitacao esta atrasada, o SLA e menor que tempo de atendimento
	 * 
	 * @param solicitacao
	 * @return true -> esta atrasada || false -> nao esta atrasada
	 * 
	 * @author gilberto.nery
	 * @since 2016-10-07
	 * 
	 */
	public boolean isSolicitacaoEmAtendimentoAtrasado(SolicitacaoServicoDTO solicitacao) {
		
		if(solicitacao == null){
			return false;
		}
		
		return calcularTempoAtraso(solicitacao) > 0 ? true : false;
		
	}

	/**
	 * Calcula o tempo de atraso em segundos
	 * 
	 * @return tempo de atraso em segundos, caso nao haja atraso retorna 0
	 */
	public int calcularTempoAtraso(SolicitacaoServicoDTO solicitacao) {
		
		Integer prazoHH = solicitacao.getPrazoHH() == null ? 0 : solicitacao.getPrazoHH(); 
		Integer prazoMM = solicitacao.getPrazoMM() == null ? 0 : solicitacao.getPrazoMM();
		Integer prazoSLA = Util.converteHHMMEmSegundos(prazoHH, prazoMM);
		
		Integer tempoAtendimentoHH = solicitacao.getTempoAtendimentoHH() == null ? 0 : solicitacao.getTempoAtendimentoHH(); 
		Integer tempoAtendimentoMM = solicitacao.getTempoAtendimentoMM() == null ? 0 : solicitacao.getTempoAtendimentoMM();
		Integer tempoAtendimentoEmSegundos = Util.converteHHMMEmSegundos(tempoAtendimentoHH, tempoAtendimentoMM);
		
		int tempoAtraso = tempoAtendimentoEmSegundos - prazoSLA;
		
		if(tempoAtraso <= 0)
			return 0;
		else
			return tempoAtraso;
	}
	
	/**
	 * <pre>
	 * Retorna o tempo de suspenso da solicitao de servio, somado ao das tarefas que no contabilizam SLA, seguindo o calendrio e apartir do 
	 * momento que se iniciou a contagem do SLA.
	 * </pre>
	 *
	 * @author euler.ramos
	 * @since 18.11.2016
	 */
	public Double obtenTempoSlaNaoContabilizado(Integer idSolicitacaoServico, Timestamp dataHoraInicioSLA) {
		return this.getControleSlaDao().obtenTempoSlaNaoContabilizado(idSolicitacaoServico, dataHoraInicioSLA);
	}
	
}