package br.com.citframework.integracao;

import java.sql.Connection;

import br.com.citframework.excecao.PersistenceException;

/**
 * Controlador de comportamento de conexes que no posuem transao
 *
 * @author bruno.ribeiro - <a href="mailto:bruno.ribeiro@centrait.com.br">bruno.ribeiro@centrait.com.br</a>
 * @since 25/08/2014
 *
 */
public interface ConnectionControler extends AutoCloseable {

    /**
     * Coloca uma conexo em estado {@code read-only}, no pode ser transacional, no executa DML de alterao
     *
     * @param readOnly
     * @throws PersistenceException
     * @since 01/09/2014
     * @see Connection#setReadOnly(boolean)
     */
    void setReadOnly(final boolean readOnly) throws PersistenceException;

    /**
     * Verifica se a conexo  ou no apenas leitura
     *
     * @return {@code true}, caso a conexo seja read only. {@code false}, caso contrrio
     * @throws PersistenceException
     * @since 01/09/2014
     */
    boolean isReadOnly() throws PersistenceException;;

    /**
     * Recupera o alias da base de dados. Normalmente, o resource JNDI do pool
     *
     * @return {@link String}
     */
    String getDataBaseAlias();

    /**
     * Seta o alias da base de dados. Normalmente, o resource JNDI do pool
     *
     * @param dataBaseAlias
     *            alias da base de dados
     */
    void setDataBaseAlias(final String dataBaseAlias);

    /**
     * Retorna a conexo relacionada ao Transaction Controler
     *
     * <p>
     * {@code IMPORTANTE}: esta trasao pode no estar com a transao iniciada ({@code {@link Connection#getAutoCommit()} == false}). Voc deve chamar {@link #start()} caso
     * queira comportamento transacional
     * <p>
     *
     * @return {@link Connection}
     */
    Connection getConnection() throws PersistenceException;

    /**
     * Commita e fecha a transao
     *
     * @throws PersistenceException
     *             caso algum problema ao fechar a transao acontea, como {@link Connection} j fechada
     * @see {@link Connection#close()}
     */
    @Override
    void close() throws PersistenceException;

    /**
     * Commit e fecha a conexo sem levantar exceces
     *
     * @return {@code true}, caso obtenha sucesso em fechar a conexo. {@code false}, caso contrrio
     * @see {@link #close()}
     */
    boolean closeQuietly();

}
