package br.com.citframework.integracao;

import java.sql.Connection;
import java.sql.Savepoint;

import br.com.citframework.excecao.PersistenceException;

public interface TransactionControler extends ConnectionControler {

    /**
     * Verifica se a traso est iniciada
     *
     * @return {@code true}, caso trasao esteja iniciada. {@code false}, caso contrrio
     */
    boolean isStarted();

    /**
     * Inicia a transao
     *
     * @throws PersistenceException
     *             caso algum problema ao iniciar a transao acontea, como {@link Connection} j fechada
     */
    void start() throws PersistenceException;

    /**
     * Commita as alteraes na transao
     *
     * @throws PersistenceException
     *             caso algum problema ao iniciar a transao acontea, como {@link Connection} j fechada
     */
    void commit() throws PersistenceException;

    /**
     * Realizao rollback de todas as alteraes ainda no commitadas na transao
     *
     * @throws PersistenceException
     *             caso algum problema ao iniciar a transao acontea, como {@link Connection} j fechada
     */
    void rollback() throws PersistenceException;

    /**
     * Realiza rollback na transao at um ponto de marcao
     *
     * @param savepoint
     *            ponto at o qual deve ser feito o rollback
     * @throws PersistenceException
     *             caso algum problema ao dar rollback na transao acontea, como {@link Connection} j fechada
     */
    void rollback(final Savepoint savepoint) throws PersistenceException;

    /**
     * Cria um {@link Savepoint} na transao
     *
     * @return {@link Savepoint}
     * @throws PersistenceException
     */
    Savepoint savepoint() throws PersistenceException;

    /**
     * Cria um {@link Savepoint} nomeado na transao
     *
     * @param name
     *            nome para identificao do ponto de marcao
     * @return {@link Savepoint}
     * @throws PersistenceException
     */
    Savepoint savepoint(final String name) throws PersistenceException;

    /**
     * Libera um {@link Savepoint} previamente marcado na linha de transaes
     *
     * @param savepoint
     *            savepoint a ser liberado na transao
     * @throws PersistenceException
     */
    void releaseSavepoint(final Savepoint savepoint) throws PersistenceException;

}
