package br.com.citframework.integracao.core;

import java.util.List;

/**
 * Representao de pgina, que  uma sublista de uma lista de objetos.
 *
 * @author bruno.ribeiro - <a href="mailto:bruno.ribeiro@centrait.com.br">bruno.ribeiro@centrait.com.br</a>
 * @since 01/10/2014
 * @see <a href="http://docs.spring.io/spring-data/commons/docs/current/api/org/springframework/data/domain/Page.html">org.springframework.data.domain.Page</a>
 */
public interface Page<T> extends Iterable<T> {

    /**
     * Retorna o nmero total de pginas
     *
     * @return {@code int}: nmero total de pginas
     */
    int getTotalPages();

    /**
     * Retorna a quantidade total de elementos
     *
     * @return {@code long}: quantidade total de elementos
     */
    long getTotalElements();

    /**
     * Retorna o nmero da {@link Page} atual.  sempre no negativa
     *
     * @return {@code int}: o nmero da {@link Page} atual
     */
    int getNumber();

    /**
     * Retorna o tamanho da {@link Page}
     *
     * @return the size of the {@link Page}
     */
    int getSize();

    /**
     * Retorna o nmero de elementos da {@link Page}
     *
     * @return {@code int} nmero de elementos da {@link Page}
     */
    int getNumberOfElements();

    /**
     * Recupera, como uma {@link List}, o contedo de uma pgina
     *
     * @return {@link List}: contedo da {@link Page}
     */
    List<T> getContent();

    /**
     * Verifica se a {@link Page} possui contedo
     *
     * @return {@code true}, caso possua contedo {@link Page}. {@code false}, caso contrrio
     */
    boolean hasContent();

    /**
     * Retorna se a {@link Page}  a primeira
     *
     * @return {@code true}, caso seja a primeira {@link Page}. {@code false}, caso contrrio
     */
    boolean isFirst();

    /**
     * Retorna se a {@link Page}  a ltima
     *
     * @return {@code true}, caso seja a ltima {@link Page}. {@code false}, caso contrrio
     */
    boolean isLast();

    /**
     * Retorna existncia de prxima pgina
     *
     * @return {@code true}, caso tenha prxima {@link Page}. {@code false}, caso contrrio
     */
    boolean hasNext();

    /**
     * Retorna existncia de pgina anterior
     *
     * @return {@code true}, caso tenha uma {@link Page} anterior. {@code false}, caso contrrio
     */
    boolean hasPrevious();

    /**
     * Recupera a {@link Pageable} para solicita a prxima {@link Page}. Pode ser {@code null} caso a {@link Page} atual seja a ltima. Usurios devem checar, chamando
     * {@link #hasNext()} antes de chamar este mtodo para ter certeza que o valor retornado no ser {@code null}
     *
     * @return {@link Pageable}
     */
    Pageable nextPageable();

    /**
     * Recupera a {@link Pageable} para solicita a {@link Page} anterior. Pode ser {@code null} caso a {@link Page} atual seja a primeira. Usurios devem checar, chamando
     * {@link #hasPrevious()} antes de chamar este mtodo para ter certeza que o valor retornado no ser {@code null}
     *
     * @return {@link Pageable}
     */
    Pageable previousPageable();

}
