package br.com.citframework.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.servlet.http.HttpServletRequest;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import br.com.centralit.citajax.html.DocumentHTML;
import br.com.centralit.citajax.util.CitAjaxUtil;
import br.com.centralit.citcorpore.bean.SmartReportDTO;
import br.com.centralit.citcorpore.bean.UsuarioDTO;
import br.com.centralit.citcorpore.metainfo.util.HashMapUtil;
import br.com.centralit.citcorpore.negocio.SmartReportService;
import br.com.centralit.citcorpore.util.CITCorporeUtil;
import br.com.centralit.citcorpore.util.Enumerados.QueryTypes;
import br.com.centralit.citcorpore.util.Enumerados.ReportTypes;
import br.com.centralit.citcorpore.util.Enumerados.TipoExibicaoRelatorio;
import br.com.centralit.citcorpore.util.WebUtil;
import br.com.centralit.citgerencial.bean.SmartReportParameterDTO;
import br.com.centralit.citgerencial.bean.SmartReportParameterOptionDTO;
import br.com.centralit.citgerencial.bean.SmartReportParameterOptionsDTO;
import br.com.citframework.dto.Usuario;
import br.com.citframework.excecao.LogicException;
import br.com.citframework.excecao.ServiceException;
import br.com.citframework.integracao.JdbcEngine;
import br.com.citframework.service.ServiceLocator;
import freemarker.template.Configuration;
import freemarker.template.Template;

/**
 * Classe utilitria para gerao dos relatrios.
 *
 * @author douglas.japiassu
 * @author renato.jesus
 */
public class UtilSmartReport {

	private static SmartReportService smartReportService;

	public static List<SmartReportParameterDTO> getReportParameters(String parametros) {
		List<SmartReportParameterDTO> smartReportParametroDTOs = new ArrayList<>();
		Document doc = null;
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();

		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			if (parametros == null) {
				return smartReportParametroDTOs;
			}
			InputStream is = new ByteArrayInputStream(parametros.getBytes());
			doc = builder.parse(is);
		} catch (Exception e) {
			e.printStackTrace();
			return smartReportParametroDTOs;
		}

		if (doc == null) {
			return smartReportParametroDTOs;
		}
		Node noItem = doc.getChildNodes().item(0);
		if (noItem == null) {
			return smartReportParametroDTOs;
		}

		if (noItem.getChildNodes() != null) {
			for (int i = 0; i < noItem.getChildNodes().getLength(); i++) {
				Node noSubItem = noItem.getChildNodes().item(i);
				if (noSubItem.getNodeName().equals("#text")) {
					continue;
				}
				if (noSubItem.getNodeName().equals("#comment")) {
					continue;
				}

				if (noSubItem.getNodeName().equalsIgnoreCase("PARAM")) {
					SmartReportParameterDTO smartReportParametroDTO = new SmartReportParameterDTO();

					NamedNodeMap map = noSubItem.getAttributes();

					smartReportParametroDTO.setType(map.getNamedItem("type").getNodeValue());

					if (map.getNamedItem("typeHTML") != null) {
						smartReportParametroDTO.setTypeHTML(map.getNamedItem("typeHTML").getNodeValue());
					}

					smartReportParametroDTO.setValue(map.getNamedItem("value").getNodeValue());
					smartReportParametroDTO.setName(map.getNamedItem("name").getNodeValue());
					smartReportParametroDTO.setDescription(map.getNamedItem("description").getNodeValue());

					// SIZE
					String size = map.getNamedItem("size").getNodeValue();
					if (size == null || size.trim().equalsIgnoreCase("")) {
						size = "0";
					}
					smartReportParametroDTO.setSize(new Integer(Integer.parseInt(size)));

					String defaultValue = null;
					if (map.getNamedItem("default") != null) {
						defaultValue = map.getNamedItem("default").getNodeValue();
					}
					if (defaultValue == null) {
						defaultValue = "";
					}

					// PARAMETROS DATE
					if (defaultValue.equalsIgnoreCase("{TODAY}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().toDate());
					}
					if (defaultValue.equalsIgnoreCase("{DAY_AGO}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().minusDays(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{WEEK_AGO}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().minusWeeks(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{MONTH_AGO}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().minusMonths(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{YEAR_AGO}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().minusYears(1).toDate());
					}

					// PARAMETROS INTEIRO/STRING
					if (defaultValue.equalsIgnoreCase("{LAST_MONTH}")) {
						defaultValue = "" + UtilDatas.getMonth(DateTime.now().minusMonths(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{CURRENT_MONTH}")) {
						defaultValue = "" + UtilDatas.getMonth(UtilDatas.getDataAtual());
					}
					if (defaultValue.equalsIgnoreCase("{LAST_YEAR}")) {
						defaultValue = "" + UtilDatas.getYear(DateTime.now().minusYears(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{CURRENT_YEAR}")) {
						defaultValue = "" + UtilDatas.getYear(UtilDatas.getDataAtual());
					}
					smartReportParametroDTO.setDefaultValue(defaultValue);

					smartReportParametroDTO.setFix(Boolean.valueOf(map.getNamedItem("fix").getNodeValue()).booleanValue());

					smartReportParametroDTO.setMandatory(Boolean.valueOf(map.getNamedItem("mandatory").getNodeValue()).booleanValue());

					// RELOAD
					if (map.getNamedItem("reload") != null) {
						if (map.getNamedItem("reload").getNodeValue() != null && !map.getNamedItem("reload").getNodeValue().equalsIgnoreCase("")) {
							smartReportParametroDTO.setReload(Boolean.valueOf(map.getNamedItem("reload").getNodeValue()).booleanValue());
						} else {
							smartReportParametroDTO.setReload(false);
						}
					} else {
						smartReportParametroDTO.setReload(false);
					}

					// SET OPTIONS PARAM (select|checkbox|radio)
					if ("select".equalsIgnoreCase(smartReportParametroDTO.getTypeHTML()) || "checkbox".equalsIgnoreCase(smartReportParametroDTO.getTypeHTML()) || "radio".equalsIgnoreCase(smartReportParametroDTO.getTypeHTML())) {
						smartReportParametroDTO.setColOptions(getRelatorioParametroOptions(noSubItem));
					}

					smartReportParametroDTOs.add(smartReportParametroDTO);
				}
			}
		}

		return smartReportParametroDTOs;
	}

	private static Collection getRelatorioParametroOptions(Node noItem) {
		if (noItem == null) {
			return null;
		}

		Collection options = new ArrayList();
		if (noItem.getChildNodes() != null) {
			for (int i = 0; i < noItem.getChildNodes().getLength(); i++) {
				Node noSubItem = noItem.getChildNodes().item(i);
				if (noSubItem.getNodeName().equals("#text")) {
					continue;
				}
				if (noSubItem.getNodeName().equals("#comment")) {
					continue;
				}

				if (noSubItem.getNodeName().equalsIgnoreCase("OPTION")) {
					NamedNodeMap map = noSubItem.getAttributes();

					SmartReportParameterOptionDTO smartReportParametroOptionDTO = new SmartReportParameterOptionDTO();
					smartReportParametroOptionDTO.setValue(map.getNamedItem("value").getNodeValue());
					smartReportParametroOptionDTO.setText(map.getNamedItem("text").getNodeValue());

					options.add(smartReportParametroOptionDTO);
				} else if (noSubItem.getNodeName().equalsIgnoreCase("OPTIONS")) {
					NamedNodeMap map = noSubItem.getAttributes();

					SmartReportParameterOptionsDTO smartReportParametroOptionsDTO = new SmartReportParameterOptionsDTO();
					String onLoad = UtilStrings.nullToVazio(map.getNamedItem("onload").getNodeValue());
					if (onLoad.equalsIgnoreCase("true")) {
						smartReportParametroOptionsDTO.setOnload(true);
					} else {
						smartReportParametroOptionsDTO.setOnload(false);
					}
					smartReportParametroOptionsDTO.setType(UtilStrings.nullToVazio(map.getNamedItem("type").getNodeValue()));
					if (smartReportParametroOptionsDTO.getType().equalsIgnoreCase("CLASS_GENERATE_SQL") || smartReportParametroOptionsDTO.getType().equalsIgnoreCase("SERVICE")) {
						smartReportParametroOptionsDTO.setClassExecute(UtilStrings.nullToVazio(noSubItem.getChildNodes().item(0).getNodeValue()).trim());
					} else {
						smartReportParametroOptionsDTO.setType("SQL");
						smartReportParametroOptionsDTO.setSql(noSubItem.getChildNodes().item(0).getNodeValue());
					}

					options.add(smartReportParametroOptionsDTO);
				}
			}
		}
		return options;
	}

	public static String getHtmlReportParameters(HttpServletRequest request, List<SmartReportParameterDTO> smartReportParametroDTOs) throws Exception {
		StringBuilder parametrosHtml = new StringBuilder();
		for (SmartReportParameterDTO smartReportParametroDTO : smartReportParametroDTOs) {
			parametrosHtml.append("<div class='params-field'>");
			parametrosHtml.append("	<label");
			if (smartReportParametroDTO.isMandatory()) {
				parametrosHtml.append(" class='campoObrigatorio'");
			}
			parametrosHtml.append(">");
			parametrosHtml.append(UtilI18N.internacionaliza(request, smartReportParametroDTO.getDescription()));
			parametrosHtml.append("</label>");

			String strValid = "";
			if (smartReportParametroDTO.isMandatory()) {
				strValid = "Required";
			}

			if (smartReportParametroDTO.getTypeHTML() != null && smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("select")) {
				// SELECT
				String strValidCompleta = "";

				if (!strValid.trim().equalsIgnoreCase("")) {
					strValidCompleta = " Valid[" + strValid + "]";
				}
				parametrosHtml.append("<select ");

				parametrosHtml.append("	name='PARAM." + smartReportParametroDTO.getName() + "' ");
				parametrosHtml.append("	id='PARAM." + smartReportParametroDTO.getName() + "' ");
				parametrosHtml.append("	class='Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + "'");
				parametrosHtml.append(">");

				if (smartReportParametroDTO.getColOptions() != null) {
					for (Object obj : smartReportParametroDTO.getColOptions()) {
						if (SmartReportParameterOptionDTO.class.isInstance(obj)) {
							SmartReportParameterOptionDTO option = (SmartReportParameterOptionDTO) obj;
							parametrosHtml.append("<option value='" + option.getValue() + "'>" + UtilI18N.internacionaliza(request, option.getText()) + "</option>");
						} else if (SmartReportParameterOptionsDTO.class.isInstance(obj)) {
							SmartReportParameterOptionsDTO options = (SmartReportParameterOptionsDTO) obj;
							if (options.isOnload()) {
								SmartReportService smartReportService = (SmartReportService) ServiceLocator.getInstance().getService(SmartReportService.class, br.com.centralit.citcorpore.util.WebUtil.getUsuarioSistema(request));
								Collection<SmartReportParameterOptionDTO> smartReportParametroOptionDTOs = smartReportService.findParametroOptions(options);
								for (SmartReportParameterOptionDTO option : smartReportParametroOptionDTOs) {
									parametrosHtml.append("<option value='" + option.getValue() + "'>" + option.getText() + "</option>");
								}
							}
						}
					}
				}
				parametrosHtml.append("</select>");
			} else if (smartReportParametroDTO.getTypeHTML() != null && smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("checkbox")) {
				// CHECKBOX
				String strValidCompleta = "";

				if (!strValid.trim().equalsIgnoreCase("")) {
					strValidCompleta = " Valid[" + strValid + "]";
				}

				if (smartReportParametroDTO.getColOptions() != null) {
					String strRetornoAux = "";
					int qtdeOpcoes = 0;

					for (Object obj : smartReportParametroDTO.getColOptions()) {
						if (SmartReportParameterOptionDTO.class.isInstance(obj)) {
							qtdeOpcoes++;
							SmartReportParameterOptionDTO option = (SmartReportParameterOptionDTO) obj;
							strRetornoAux += "<div class='checkbox'>";
							strRetornoAux += "	<label>";
							strRetornoAux += "		<input " + "					type='checkbox' " + "					name='PARAM." + smartReportParametroDTO.getName() + "' " + "					id='PARAM."
									+ smartReportParametroDTO.getName() + "' " + "					class='Description[" + smartReportParametroDTO.getDescription() + "] " + strValidCompleta + "' " + "					value='"
									+ option.getValue() + "'" + "				/>" + option.getText();
							strRetornoAux += "	</label>";
							strRetornoAux += "</div>";
						} else if (SmartReportParameterOptionsDTO.class.isInstance(obj)) {
							SmartReportParameterOptionsDTO options = (SmartReportParameterOptionsDTO) obj;

							if (options.isOnload()) {
								SmartReportService smartReportService = (SmartReportService) ServiceLocator.getInstance().getService(SmartReportService.class, br.com.centralit.citcorpore.util.WebUtil.getUsuarioSistema(request));
								Collection<SmartReportParameterOptionDTO> smartReportParametroOptionDTOs = smartReportService.findParametroOptions(options);
								for (SmartReportParameterOptionDTO option : smartReportParametroOptionDTOs) {
									qtdeOpcoes++;

									strRetornoAux += "<div class='checkbox'>";
									strRetornoAux += "	<label>";
									strRetornoAux += "		<input " + "					type='checkbox' " + "					name='PARAM." + smartReportParametroDTO.getName() + "' " + "					id='PARAM."
											+ smartReportParametroDTO.getName() + "' " + "					class='Description[" + smartReportParametroDTO.getDescription() + "] " + strValidCompleta + "' " + "					value='"
											+ option.getValue() + "'" + "				/>" + option.getText();
									strRetornoAux += "	</label>";
									strRetornoAux += "</div>";
								}
							}
						}
					}

					if (!strRetornoAux.equalsIgnoreCase("")) {
						if (qtdeOpcoes > 5) {
							parametrosHtml.append("<div style='height:100px; overflow:auto; border: 1px solid black'>" + strRetornoAux + "</div>");
						} else {
							parametrosHtml.append(strRetornoAux);
						}
					}
				}
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.sql.Date")) {
				// DATE
				if (!UtilStrings.isNullOrEmpty(strValid)) {
					strValid += ",";
				}
				strValid += "Date";

				String fieldFilled = "";
				String uuid = UUID.randomUUID().toString();

				parametrosHtml.append("<input ");
				parametrosHtml.append("	type='text'");
				parametrosHtml.append("	maxlength='" + smartReportParametroDTO.getSize() + "'");
				parametrosHtml.append("	name='PARAM." + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append("	id='PARAM." + smartReportParametroDTO.getName() + "_" + uuid + "'");
				if (smartReportParametroDTO.isMandatory() && !UtilStrings.isNullOrEmpty(smartReportParametroDTO.getDefaultValue())) {
					parametrosHtml.append(" value='" + smartReportParametroDTO.getDefaultValue() + "' ");
					fieldFilled = "filled";
				}
				parametrosHtml.append(" class='Format[Date] Description[" + smartReportParametroDTO.getDescription() + "] Valid[" + strValid + "] " + fieldFilled + " datepicker' ");
				parametrosHtml.append("/>");
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.lang.Integer")) {
				// INTEGER
				String strValidCompleta = "";
				if (!strValid.trim().equalsIgnoreCase("")) {
					strValidCompleta = " Valid[" + strValid + "]";
				}

				parametrosHtml.append("<input ");
				parametrosHtml.append(" type='text'");
				parametrosHtml.append(" size='" + smartReportParametroDTO.getSize() + "'");
				parametrosHtml.append(" maxlength='" + smartReportParametroDTO.getSize() + "'");

				String prefix = "PARAM.";
				if (smartReportParametroDTO.getName().equalsIgnoreCase("specificID")) {
					prefix = "";
				}

				parametrosHtml.append(" name='" + prefix + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append(" id='" + prefix + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append(" class='Format[Numero] Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + "'");
				parametrosHtml.append("/>");
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.lang.Double")) {
				// DOUBLE
				String strValidCompleta = "";
				if (!strValid.trim().equalsIgnoreCase("")) {
					strValidCompleta = " Valid[" + strValid + "]";
				}

				parametrosHtml.append("<input ");
				parametrosHtml.append(" type='text'");
				parametrosHtml.append(" maxlength='" + smartReportParametroDTO.getSize() + "'");
				parametrosHtml.append(" name='PARAM." + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append(" id='PARAM." + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append(" class='Format[Money] Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + "'");
				parametrosHtml.append("/>");
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.lang.String")) {
				// STRING
				if (smartReportParametroDTO.getTypeHTML() == null || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("") || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("text")) {
					String strValidCompleta = "";
					if (!strValid.trim().equalsIgnoreCase("")) {
						strValidCompleta = " Valid[" + strValid + "]";
					}

					parametrosHtml.append("<input ");
					parametrosHtml.append(" type='text'");
					parametrosHtml.append(" maxlength='" + smartReportParametroDTO.getSize() + "'");
					parametrosHtml.append(" name='PARAM." + smartReportParametroDTO.getName() + "'");
					parametrosHtml.append(" id='PARAM." + smartReportParametroDTO.getName() + "'");
					parametrosHtml.append(" class='Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + "'");
					parametrosHtml.append("/>");
				}
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.lang.StringBuilder")) {
				// TEXTAREA
				if (smartReportParametroDTO.getTypeHTML() == null || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("") || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("text")
						|| smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("textarea")) {
					String strValidCompleta = "";
					if (!strValid.trim().equalsIgnoreCase("")) {
						strValidCompleta = " Valid[" + strValid + "]";
					}

					parametrosHtml.append("<textarea ");
					parametrosHtml.append(" rows='" + smartReportParametroDTO.getSize() + "'");
					parametrosHtml.append(" cols='70'");
					parametrosHtml.append(" name='PARAM." + smartReportParametroDTO.getName() + "'");
					parametrosHtml.append(" id='PARAM." + smartReportParametroDTO.getName() + "'");
					parametrosHtml.append(" class='Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + "'");
					parametrosHtml.append(">");
					parametrosHtml.append("</textarea>");
				}
			}

			parametrosHtml.append("</div>");
		}

		return parametrosHtml.toString();
	}

	public static String getHtmlReportParametersSmartDecisions(HttpServletRequest request, List<SmartReportParameterDTO> smartReportParametroDTOs) throws Exception {
		StringBuilder parametrosHtml = new StringBuilder();
		for (SmartReportParameterDTO smartReportParametroDTO : smartReportParametroDTOs) {
			parametrosHtml.append("<div class='col-md-6 params-field'>");
			parametrosHtml.append("	<label");
			if (smartReportParametroDTO.isMandatory()) {
				parametrosHtml.append(" class='campoObrigatorio'");
			}
			parametrosHtml.append(">");
			parametrosHtml.append(UtilI18N.internacionaliza(request, smartReportParametroDTO.getDescription()));
			parametrosHtml.append("</label>");

			String strValid = "";
			if (smartReportParametroDTO.isMandatory()) {
				strValid = "Required";
			}

			if (smartReportParametroDTO.getTypeHTML() != null && smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("select")) {
				// SELECT
				String strValidCompleta = "";

				if (!strValid.trim().equalsIgnoreCase("")) {
					strValidCompleta = " Valid[" + strValid + "]";
				}
				parametrosHtml.append("<select ");

				parametrosHtml.append("	name='PARAM." + smartReportParametroDTO.getName() + "' ");
				parametrosHtml.append("	id='PARAM." + smartReportParametroDTO.getName() + "' ");
				parametrosHtml.append("	class='Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + " form-control'");
				parametrosHtml.append(">");

				if (smartReportParametroDTO.getColOptions() != null) {
					for (Object obj : smartReportParametroDTO.getColOptions()) {
						if (SmartReportParameterOptionDTO.class.isInstance(obj)) {
							SmartReportParameterOptionDTO option = (SmartReportParameterOptionDTO) obj;
							parametrosHtml.append("<option value='" + option.getValue() + "'>" + UtilI18N.internacionaliza(request, option.getText()) + "</option>");
						} else if (SmartReportParameterOptionsDTO.class.isInstance(obj)) {
							SmartReportParameterOptionsDTO options = (SmartReportParameterOptionsDTO) obj;
							if (options.isOnload()) {
								SmartReportService smartReportService = (SmartReportService) ServiceLocator.getInstance().getService(SmartReportService.class, br.com.centralit.citcorpore.util.WebUtil.getUsuarioSistema(request));
								Collection<SmartReportParameterOptionDTO> smartReportParametroOptionDTOs = smartReportService.findParametroOptions(options);
								for (SmartReportParameterOptionDTO option : smartReportParametroOptionDTOs) {
									parametrosHtml.append("<option value='" + option.getValue() + "'>" + option.getText() + "</option>");
								}
							}
						}
					}
				}
				parametrosHtml.append("</select>");
			} else if (smartReportParametroDTO.getTypeHTML() != null && smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("checkbox")) {
				// CHECKBOX
				String strValidCompleta = "";

				if (!strValid.trim().equalsIgnoreCase("")) {
					strValidCompleta = " Valid[" + strValid + "]";
				}

				if (smartReportParametroDTO.getColOptions() != null) {
					String strRetornoAux = "";
					int qtdeOpcoes = 0;

					for (Object obj : smartReportParametroDTO.getColOptions()) {
						if (SmartReportParameterOptionDTO.class.isInstance(obj)) {
							qtdeOpcoes++;
							SmartReportParameterOptionDTO option = (SmartReportParameterOptionDTO) obj;
							strRetornoAux += "<div class='checkbox'>";
							strRetornoAux += "	<label>";
							strRetornoAux += "		<input " + "					type='checkbox' " + "					name='PARAM." + smartReportParametroDTO.getName() + "' " + "					id='PARAM." + smartReportParametroDTO.getName() + "' " + "					class='Description[" + smartReportParametroDTO.getDescription() + "] " + strValidCompleta + "' "
									+ "					value='" + option.getValue() + "'" + "				/>" + option.getText();
							strRetornoAux += "	</label>";
							strRetornoAux += "</div>";
						} else if (SmartReportParameterOptionsDTO.class.isInstance(obj)) {
							SmartReportParameterOptionsDTO options = (SmartReportParameterOptionsDTO) obj;

							if (options.isOnload()) {
								SmartReportService smartReportService = (SmartReportService) ServiceLocator.getInstance().getService(SmartReportService.class, br.com.centralit.citcorpore.util.WebUtil.getUsuarioSistema(request));
								Collection<SmartReportParameterOptionDTO> smartReportParametroOptionDTOs = smartReportService.findParametroOptions(options);
								for (SmartReportParameterOptionDTO option : smartReportParametroOptionDTOs) {
									qtdeOpcoes++;

									strRetornoAux += "<div class='checkbox'>";
									strRetornoAux += "	<label>";
									strRetornoAux += "		<input " + "					type='checkbox' " + "					name='PARAM." + smartReportParametroDTO.getName() + "' " + "					id='PARAM." + smartReportParametroDTO.getName() + "' " + "					class='Description[" + smartReportParametroDTO.getDescription() + "] " + strValidCompleta + "' "
											+ "					value='" + option.getValue() + "'" + "				/>" + option.getText();
									strRetornoAux += "	</label>";
									strRetornoAux += "</div>";
								}
							}
						}
					}

					if (!strRetornoAux.equalsIgnoreCase("")) {
						if (qtdeOpcoes > 5) {
							parametrosHtml.append("<div style='height:100px; overflow:auto; border: 1px solid black'>" + strRetornoAux + "</div>");
						} else {
							parametrosHtml.append(strRetornoAux);
						}
					}
				}
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.sql.Date")) {
				// DATE
				if (!UtilStrings.isNullOrEmpty(strValid)) {
					strValid += ",";
				}
				strValid += "Date";

				String fieldFilled = "";
				String uuid = UUID.randomUUID().toString();

				parametrosHtml.append("<input ");
				parametrosHtml.append("	type='text'");
				parametrosHtml.append("	maxlength='" + smartReportParametroDTO.getSize() + "'");
				parametrosHtml.append("	name='PARAM." + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append("	id='PARAM." + smartReportParametroDTO.getName() + "_" + uuid + "'");
				parametrosHtml.append("	ng-click='open(\"" + uuid + "\")' uib-datepicker-popup='{{format}}' ng-model='dt[\"" + uuid + "\"]' is-open='elements[\"" + uuid + "\"].opened' datepicker-options='dateOptions' ng-required='true' alt-input-formats='altInputFormats' current-text='" + UtilI18N.internacionaliza(request, "citcorpore.comum.hoje") + "' clear-text='" + UtilI18N.internacionaliza(request, "citcorpore.ui.botao.rotulo.Limpar") + "' close-text='" + UtilI18N.internacionaliza(request, "Citsmart.comum.fechar") + "'");
				if (!UtilStrings.isNullOrEmpty(smartReportParametroDTO.getDefaultValue())) {
					List<String> data = Arrays.asList(smartReportParametroDTO.getDefaultValue().split("/"));

					if (data.size() == 3) {
						Collections.reverse(data);
						parametrosHtml.append(" ng-init='setInitialDate(\"" + uuid + "\", " + StringUtils.join(data, ",") + ")' ");
					} else {
						parametrosHtml.append(" ng-init='setInitialDate(\"" + uuid + "\", " + smartReportParametroDTO.getDefaultValue() + ")' ");
					}
					parametrosHtml.append(" value='" + smartReportParametroDTO.getDefaultValue() + "' ");
					fieldFilled = "filled";
				}
				parametrosHtml.append(" class='Description[" + smartReportParametroDTO.getDescription() + "] Valid[" + strValid + "] " + fieldFilled + " form-control' ");
				parametrosHtml.append("/>");
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.lang.Integer")) {
				// INTEGER
				String strValidCompleta = "";
				if (!strValid.trim().equalsIgnoreCase("")) {
					strValidCompleta = " Valid[" + strValid + "]";
				}

				parametrosHtml.append("<input ");
				parametrosHtml.append(" type='text'");
				parametrosHtml.append(" size='" + smartReportParametroDTO.getSize() + "'");
				parametrosHtml.append(" maxlength='" + smartReportParametroDTO.getSize() + "'");

				String prefix = "PARAM.";
				if (smartReportParametroDTO.getName().equalsIgnoreCase("specificID")) {
					prefix = "";
				}

				parametrosHtml.append(" name='" + prefix + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append(" id='" + prefix + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append(" class='Format[Numero] Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + " form-control'");
				parametrosHtml.append("/>");
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.lang.Double")) {
				// DOUBLE
				String strValidCompleta = "";
				if (!strValid.trim().equalsIgnoreCase("")) {
					strValidCompleta = " Valid[" + strValid + "]";
				}

				parametrosHtml.append("<input ");
				parametrosHtml.append(" type='text'");
				parametrosHtml.append(" maxlength='" + smartReportParametroDTO.getSize() + "'");
				parametrosHtml.append(" name='PARAM." + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append(" id='PARAM." + smartReportParametroDTO.getName() + "'");
				parametrosHtml.append(" class='Format[Money] Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + " form-control'");
				parametrosHtml.append("/>");
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.lang.String")) {
				// STRING
				if (smartReportParametroDTO.getTypeHTML() == null || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("") || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("text")) {
					String strValidCompleta = "";
					if (!strValid.trim().equalsIgnoreCase("")) {
						strValidCompleta = " Valid[" + strValid + "]";
					}

					parametrosHtml.append("<input ");
					parametrosHtml.append(" type='text'");
					parametrosHtml.append(" maxlength='" + smartReportParametroDTO.getSize() + "'");
					parametrosHtml.append(" name='PARAM." + smartReportParametroDTO.getName() + "'");
					parametrosHtml.append(" id='PARAM." + smartReportParametroDTO.getName() + "'");
					parametrosHtml.append(" class='Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + " form-control'");
					parametrosHtml.append("/>");
				}
			} else if (smartReportParametroDTO.getType().equalsIgnoreCase("java.lang.StringBuilder")) {
				// TEXTAREA
				if (smartReportParametroDTO.getTypeHTML() == null || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("") || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("text") || smartReportParametroDTO.getTypeHTML().equalsIgnoreCase("textarea")) {
					String strValidCompleta = "";
					if (!strValid.trim().equalsIgnoreCase("")) {
						strValidCompleta = " Valid[" + strValid + "]";
					}

					parametrosHtml.append("<textarea ");
					parametrosHtml.append(" rows='" + smartReportParametroDTO.getSize() + "'");
					parametrosHtml.append(" cols='70'");
					parametrosHtml.append(" name='PARAM." + smartReportParametroDTO.getName() + "'");
					parametrosHtml.append(" id='PARAM." + smartReportParametroDTO.getName() + "'");
					parametrosHtml.append(" class='Description[" + smartReportParametroDTO.getDescription() + "]" + strValidCompleta + " form-control'");
					parametrosHtml.append(">");
					parametrosHtml.append("</textarea>");
				}
			}

			parametrosHtml.append("</div>");
		}

		return parametrosHtml.toString();
	}

	/**
	 * @param document
	 * @param request
	 * @param response
	 * @throws Exception
	 *
	 * @author douglas.japiassu
	 * @since 20.09.2016
	 */
	public static void processReport(DocumentHTML document, HttpServletRequest request) throws Exception {
		UsuarioDTO usuario = WebUtil.getUsuario(request);
		if (!WebUtil.validarSeUsuarioEstaNaSessao(request, document)) {
			return;
		}

		SmartReportDTO smartReportDTO = (SmartReportDTO) document.getBean();

		if (smartReportDTO != null && smartReportDTO.getTipoConsulta() != null) {
			if (QueryTypes.JSP.getId().equals(smartReportDTO.getTipoConsulta()) && UtilStrings.isNotVazio(smartReportDTO.getJsp())) {
				processJSP(document, request, usuario, smartReportDTO);
			} else {
				List<Object[]> listValues = new ArrayList<>();
				List<String> listAlias = new ArrayList<String>();
				StringBuilder htmlResult = new StringBuilder();

				try {
					if (QueryTypes.SQL.getId().equals(smartReportDTO.getTipoConsulta()) && UtilStrings.isNotVazio(smartReportDTO.getSql())) {
						executeQuery(request, smartReportDTO, listValues, listAlias);
					} else if (QueryTypes.RHINOSCRIPT.getId().equals(smartReportDTO.getTipoConsulta()) && UtilStrings.isNotVazio(smartReportDTO.getScript())) {
						executeRhinoScript(request, usuario, smartReportDTO, listValues, listAlias, htmlResult);
					}

					if (smartReportDTO.getTipoRelatorio() != null) {
						if (ReportTypes.HTML_RETURN.getId().equals(smartReportDTO.getTipoRelatorio())) {
							document.executeScript("UtilSmartReport.buildSample(" + smartReportDTO.getIdSmartReport() + ", '" + smartReportDTO.getTipoRelatorio() + "', '" + StringEscapeUtils.escapeJavaScript(htmlResult.toString()) + "');");
						} else if (ReportTypes.LINE_CHART.getId().equals(smartReportDTO.getTipoRelatorio()) || ReportTypes.PIE_CHART.getId().equals(smartReportDTO.getTipoRelatorio())
								|| ReportTypes.BAR_CHART.getId().equals(smartReportDTO.getTipoRelatorio())) {
							processChartReports(document, request, smartReportDTO, listValues);
						} else if (ReportTypes.TEMPLATE.getId().equals(smartReportDTO.getTipoRelatorio())) {
							processTemplateReport(document, request, usuario, smartReportDTO, listValues);
						} else if (ReportTypes.CROSS_REFERENCES.getId().equals(smartReportDTO.getTipoRelatorio())) {
							processCrossReferencesReport(document, smartReportDTO, listValues, listAlias);
						} else if (ReportTypes.DATA_TABLES.getId().equals(smartReportDTO.getTipoRelatorio())) {
							processDataTableReport(document, smartReportDTO, listValues, listAlias);
						}
					} else {
						document.executeScript("UtilSmartReport.reportError(" + smartReportDTO.getIdSmartReport() + ", 'construtorconsultas.naoFoiPossivelConstruirRelatorio')");
						// document.executeScript("notificacaoErro('', '" + UtilI18N.internacionaliza(request, "construtorconsultas.naoFoiPossivelConstruirRelatorio") + ".');");
					}
				} catch (Exception e) {
					e.printStackTrace();
					document.executeScript("UtilSmartReport.reportError(" + smartReportDTO.getIdSmartReport() + ", 'construtorconsultas.naoFoiPossivelConstruirRelatorio')");
					// document.executeScript("notificacaoErro('', '" + UtilI18N.internacionaliza(request, "construtorconsultas.naoFoiPossivelConstruirRelatorio") + ".');");
					// document.executeScript("UtilSmartReport.buildSample(" + smartReportDTO.getIdSmartReport() + ", '" + smartReportDTO.getTipoRelatorio() + "', '');");
				}
			}
		}
	}

	/**
	 * @param document
	 * @param request
	 * @param usuario
	 * @param smartReportDTO
	 * @throws IOException
	 *
	 * @author douglas.japiassu
	 * @since 20.09.2016
	 */
	private static void processJSP(DocumentHTML document, HttpServletRequest request, UsuarioDTO usuario, SmartReportDTO smartReportDTO) throws IOException {
		Integer specificID = null;

		if (TipoExibicaoRelatorio.ESPECIFICO.getIdentificador().equalsIgnoreCase(smartReportDTO.getTipoExibicao())) {
			try {
				specificID = Integer.parseInt(request.getParameter("specificID"));
			} catch (Exception e) {

			}
		}

		smartReportDTO.setJsp(filtraConteudo(smartReportDTO.getJsp()));

		File f = new File(CITCorporeUtil.CAMINHO_REAL_APP + "/jspEmbedded");
		if (!f.exists()) {
			f.mkdirs();
		}

		f = new File(CITCorporeUtil.CAMINHO_REAL_APP + "/jspEmbedded/" + usuario.getIdUsuario());
		if (!f.exists()) {
			f.mkdirs();
		}

		UtilTratamentoArquivos.geraFileTxtFromString(CITCorporeUtil.CAMINHO_REAL_APP + "/jspEmbedded/" + usuario.getIdUsuario() + "/jsp_" + smartReportDTO.getIdentificacao() + "_process.jsp", smartReportDTO.getJsp());

		document.executeScript("UtilSmartReport.buildJSP(" + smartReportDTO.getIdSmartReport() + ", '" + CITCorporeUtil.CAMINHO_SERVIDOR + request.getContextPath() + "/jspEmbedded/" + usuario.getIdUsuario() + "/jsp_"
				+ smartReportDTO.getIdentificacao() + "_process.jsp" + "', "
				+ specificID + ");");
	}

	private static String filtraConteudo(String content) {
		content = removeTag(content, "noCache.jsp");
		content = removeTag(content, "tags/cit");
		content = removeTag(content, "tags/i18n");

		int index = content.indexOf("br.com.citframework.util.UtilI18N");
		if (index == -1) {
			content = "<%@page import=\"br.com.citframework.util.UtilI18N\"%>" + content;
		}

		content = content.replaceAll("i18n:message", "fmt:message");
		content = content.replaceAll("caminho_real_app", "CAMINHO_REAL_APP");

		return content;
	}

	private static String removeTag(String jsp, String tag) {
		int index = jsp.indexOf(tag);
		if (index != -1) {
			String antes = jsp.substring(0, index + 1), depois = jsp.substring(index, jsp.length());
			int inicio = antes.lastIndexOf("<%"), fim = depois.indexOf("%>") + 2;
			jsp = jsp.substring(0, inicio) + jsp.substring(index + fim, jsp.length());
		}

		return jsp;
	}

	/**
	 * @param smartReportDTO
	 * @param listValues
	 * @param listAlias
	 * @throws Exception
	 * @throws ServiceException
	 *
	 * @author douglas.japiassu
	 * @param request
	 * @since 20.09.2016
	 */
	public static void executeQuery(HttpServletRequest request, SmartReportDTO smartReportDTO, List<Object[]> listValues, List<String> listAlias) throws Exception, ServiceException {
		Collection<String> parametrosUtilizadosNoSQL = new ArrayList<String>();
		smartReportDTO.setSql(trataSQL(request, smartReportDTO.getSql(), WebUtil.getUsuarioSistema(request), parametrosUtilizadosNoSQL));
		HashMap<String, Object> hashParametros = getParametrosInformados(request);
		List listParms = trataParameters(hashParametros, parametrosUtilizadosNoSQL, getParametros(smartReportDTO.getParametros()));

		List<Map<String, Object>> listMapAliasValues = getSmartReportService().executeQuery(smartReportDTO, listParms.toArray());
		Map<String, Object> map = Collections.emptyMap();

		for (int i = 0; i < listMapAliasValues.size(); i++) {
			map = listMapAliasValues.get(i);
			List<Object> objs = new ArrayList<Object>();
			for (Map.Entry<String, Object> entry : map.entrySet()) {
				if (i == 0) {
					listAlias.add(entry.getKey());
				}
				objs.add(entry.getValue());
			}
			listValues.add(objs.toArray());
		}
	}

	/**
	 *
	 * @param request
	 * @param smartReportDTO
	 * @return
	 * @throws Exception
	 * @throws ServiceException
	 *
	 * @author douglas.japiassu
	 * @since 04.10.2016
	 */
	public static List<String> getResultColumns(HttpServletRequest request, SmartReportDTO smartReportDTO) throws Exception, ServiceException {
		Collection<String> parametrosUtilizadosNoSQL = new ArrayList<String>();
		smartReportDTO.setSql(trataSQL(request, smartReportDTO.getSql(), WebUtil.getUsuarioSistema(request), parametrosUtilizadosNoSQL));
		HashMap<String, Object> hashParametros = getParametrosInformados(request);
		List listParms = trataParameters(hashParametros, parametrosUtilizadosNoSQL, getParametros(smartReportDTO.getParametros()));

		return getSmartReportService().getResultColumns(smartReportDTO, listParms.toArray());
	}

	private static String trataSQL(HttpServletRequest request, String sql, Usuario usuario, Collection<String> parametrosUtilizadosNoSQL) {
		sql = sql.replaceAll("\\{IDEMPRESA\\}", "" + usuario.getIdEmpresa());
		sql = sql.replaceAll("\\{DATAATUAL\\}", "'" + UtilDatas.dateToSTR(UtilDatas.getDataAtual()) + "'");

		try {
			sql = sql.replaceAll("\\{SPECIFIC_ID\\}", request.getParameter("specificID"));
		} catch (Exception e) {
			sql = sql.replaceAll("\\{SPECIFIC_ID\\}", "0");
		}

		boolean continua = true;
		while (continua) {
			int beginIndex = sql.indexOf("{PARAM");
			if (beginIndex >= 0) {
				int endIndex = sql.indexOf("}");
				String nomeParametro = sql.substring(beginIndex, endIndex);
				nomeParametro = nomeParametro.replaceAll("\\{", "");

				sql = sql.replaceFirst("\\{" + nomeParametro + "\\}", "?");

				parametrosUtilizadosNoSQL.add(nomeParametro);
			} else {
				continua = false;
			}
		}

		return sql;
	}

	private static List trataParameters(HashMap<String, Object> hsmParms, Collection<String> colParmsUtilizadosNoSQL, Collection<SmartReportParameterDTO> colDefinicaoParametros) {
		List lstRetorno = new ArrayList();
		if (colParmsUtilizadosNoSQL == null || colParmsUtilizadosNoSQL.size() == 0) {
			return lstRetorno;
		}

		for (Iterator<String> it = colParmsUtilizadosNoSQL.iterator(); it.hasNext();) {
			String nameParm = it.next();
			String type = getParameterType(colDefinicaoParametros, nameParm);

			String valor = (String) hsmParms.get(nameParm);

			if (type.equalsIgnoreCase("java.sql.Date")) {
				Date data = null;
				try {
					data = UtilDatas.strToSQLDate(valor);
				} catch (LogicException e) {
					e.printStackTrace();
				}
				lstRetorno.add(data);
			} else if (type.equalsIgnoreCase("java.lang.Integer")) {
				Integer intAux = null;
				if (valor == null) {
					intAux = new Integer(0);
				} else {
					intAux = new Integer(valor);
				}
				lstRetorno.add(intAux);
			} else if (type.equalsIgnoreCase("java.lang.Double")) {
				Double duplo;

				String aux = valor;
				aux = aux.replaceAll("\\.", "");
				aux = aux.replaceAll("\\,", "\\.");

				duplo = new Double(Double.parseDouble(aux));
				lstRetorno.add(duplo);
			} else if (type.equalsIgnoreCase("java.lang.String")) {
				lstRetorno.add(valor);
			}
		}

		return lstRetorno;
	}

	private static String getParameterType(Collection<SmartReportParameterDTO> colDefinicaoParametros, String nameParm) {
		SmartReportParameterDTO parameter = getParameter(colDefinicaoParametros, nameParm);

		if (parameter != null) {
			return parameter.getType();
		}
		return "";
	}

	private static SmartReportParameterDTO getParameter(Collection<SmartReportParameterDTO> colDefinicaoParametros, String nameParm) {
		if (colDefinicaoParametros == null) {
			return null;
		}

		for (Iterator<SmartReportParameterDTO> it = colDefinicaoParametros.iterator(); it.hasNext();) {
			SmartReportParameterDTO smartReportParametroDTO = it.next();
			String nomeParmAux = "PARAM." + smartReportParametroDTO.getName().trim();
			if (nomeParmAux.equalsIgnoreCase(nameParm)) {
				return smartReportParametroDTO;
			}
		}
		return null;
	}

	private static HashMap<String, Object> getParametrosInformados(HttpServletRequest request) {
		Enumeration<String> parameters = request.getParameterNames();
		HashMap<String, Object> hashRetorno = new HashMap<String, Object>();
		String[] aux;
		while (parameters.hasMoreElements()) {
			String nameElement = parameters.nextElement();

			if (nameElement.startsWith("PARAM.")) {
				String[] strValores = request.getParameterValues(nameElement);
				if (strValores.length == 0 || strValores.length == 1) {
					String value = request.getParameter(nameElement);
					hashRetorno.put(nameElement, value);
				} else {
					aux = new String[strValores.length];
					for (int i = 0; i < strValores.length; i++) {
						aux[i] = strValores[i];
					}
					hashRetorno.put(nameElement, aux);
				}
			}
		}

		return hashRetorno;
	}

	private static List<SmartReportParameterDTO> getParametros(String parametros) {
		Document doc = null;
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			if (parametros == null) {
				return null;
			}
			InputStream is = new ByteArrayInputStream(parametros.getBytes());
			doc = builder.parse(is);
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
		if (doc == null) {
			return null;
		}
		Node noItem = doc.getChildNodes().item(0);
		if (noItem == null) {
			return null;
		}
		List<SmartReportParameterDTO> colParameters = new ArrayList<SmartReportParameterDTO>();
		SmartReportParameterDTO parametro;
		if (noItem.getChildNodes() != null) {
			for (int i = 0; i < noItem.getChildNodes().getLength(); i++) {
				Node noSubItem = noItem.getChildNodes().item(i);
				if (noSubItem.getNodeName().equals("#text")) {
					continue;
				}
				if (noSubItem.getNodeName().equals("#comment")) {
					continue;
				}

				if (noSubItem.getNodeName().equalsIgnoreCase("PARAM")) {
					parametro = new SmartReportParameterDTO();

					NamedNodeMap map = noSubItem.getAttributes();

					parametro.setType(map.getNamedItem("type").getNodeValue());
					if (map.getNamedItem("typeHTML") != null) {
						parametro.setTypeHTML(map.getNamedItem("typeHTML").getNodeValue());
					}
					parametro.setValue(map.getNamedItem("value").getNodeValue());
					parametro.setName(map.getNamedItem("name").getNodeValue());
					parametro.setDescription(map.getNamedItem("description").getNodeValue());

					String size = map.getNamedItem("size").getNodeValue();
					if (size == null || size.trim().equalsIgnoreCase("")) {
						size = "0";
					}
					parametro.setSize(new Integer(Integer.parseInt(size)));

					String defaultValue = null;
					if (map.getNamedItem("default") != null) {
						defaultValue = map.getNamedItem("default").getNodeValue();
					}
					if (defaultValue == null) {
						defaultValue = "";
					}

					// PARAMETROS DATE
					if (defaultValue.equalsIgnoreCase("{TODAY}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().toDate());
					}
					if (defaultValue.equalsIgnoreCase("{DAY_AGO}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().minusDays(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{WEEK_AGO}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().minusWeeks(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{MONTH_AGO}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().minusMonths(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{YEAR_AGO}")) {
						defaultValue = UtilDatas.dateToSTR(DateTime.now().minusYears(1).toDate());
					}

					// PARAMETROS INTEIRO/STRING
					if (defaultValue.equalsIgnoreCase("{LAST_MONTH}")) {
						defaultValue = "" + UtilDatas.getMonth(DateTime.now().minusMonths(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{CURRENT_MONTH}")) {
						defaultValue = "" + UtilDatas.getMonth(UtilDatas.getDataAtual());
					}
					if (defaultValue.equalsIgnoreCase("{LAST_YEAR}")) {
						defaultValue = "" + UtilDatas.getYear(DateTime.now().minusYears(1).toDate());
					}
					if (defaultValue.equalsIgnoreCase("{CURRENT_YEAR}")) {
						defaultValue = "" + UtilDatas.getYear(UtilDatas.getDataAtual());
					}
					parametro.setDefaultValue(defaultValue);

					parametro.setFix(Boolean.valueOf(map.getNamedItem("fix").getNodeValue()).booleanValue());
					parametro.setMandatory(Boolean.valueOf(map.getNamedItem("mandatory").getNodeValue()).booleanValue());
					if (map.getNamedItem("reload") != null) {
						if (map.getNamedItem("reload").getNodeValue() != null && !map.getNamedItem("reload").getNodeValue().equalsIgnoreCase("")) {
							parametro.setReload(Boolean.valueOf(map.getNamedItem("reload").getNodeValue()).booleanValue());
						} else {
							parametro.setReload(false);
						}
					} else {
						parametro.setReload(false);
					}

					if ("select".equalsIgnoreCase(parametro.getTypeHTML()) || "checkbox".equalsIgnoreCase(parametro.getTypeHTML()) || "radio".equalsIgnoreCase(parametro.getTypeHTML())) {
						parametro.setColOptions(getParametrosOptions(noSubItem));
					}

					colParameters.add(parametro);
				}
			}
		}
		return colParameters;
	}

	public static Collection getParametrosOptions(Node noItem) {
		if (noItem == null) {
			return null;
		}

		Collection colRetorno = new ArrayList();
		if (noItem.getChildNodes() != null) {
			for (int i = 0; i < noItem.getChildNodes().getLength(); i++) {
				Node noSubItem = noItem.getChildNodes().item(i);
				if (noSubItem.getNodeName().equals("#text")) {
					continue;
				}
				if (noSubItem.getNodeName().equals("#comment")) {
					continue;
				}

				if (noSubItem.getNodeName().equalsIgnoreCase("OPTION")) {
					NamedNodeMap map = noSubItem.getAttributes();

					SmartReportParameterOptionDTO smartReportParametroOptionDTO = new SmartReportParameterOptionDTO();
					smartReportParametroOptionDTO.setValue(map.getNamedItem("value").getNodeValue());
					smartReportParametroOptionDTO.setText(map.getNamedItem("text").getNodeValue());

					colRetorno.add(smartReportParametroOptionDTO);
				}

				if (noSubItem.getNodeName().equalsIgnoreCase("OPTIONS")) {
					NamedNodeMap map = noSubItem.getAttributes();

					SmartReportParameterOptionsDTO smartReportParametroOptionsDTO = new SmartReportParameterOptionsDTO();
					String onLoad = UtilStrings.nullToVazio(map.getNamedItem("onload").getNodeValue());
					if (onLoad.equalsIgnoreCase("true")) {
						smartReportParametroOptionsDTO.setOnload(true);
					} else {
						smartReportParametroOptionsDTO.setOnload(false);
					}
					smartReportParametroOptionsDTO.setType(UtilStrings.nullToVazio(map.getNamedItem("type").getNodeValue()));
					if (smartReportParametroOptionsDTO.getType().equalsIgnoreCase("CLASS_GENERATE_SQL") || smartReportParametroOptionsDTO.getType().equalsIgnoreCase("SERVICE")) {
						smartReportParametroOptionsDTO.setClassExecute(UtilStrings.nullToVazio(noSubItem.getChildNodes().item(0).getNodeValue()).trim());
					} else {
						smartReportParametroOptionsDTO.setType("SQL");
						smartReportParametroOptionsDTO.setSql(noSubItem.getChildNodes().item(0).getNodeValue());
					}

					colRetorno.add(smartReportParametroOptionsDTO);
				}
			}
		}
		return colRetorno;
	}

	/**
	 * @param request
	 * @param usuario
	 * @param smartReportDTO
	 * @param listValues
	 * @param listAlias
	 * @param htmlResult
	 *
	 * @author douglas.japiassu
	 * @since 20.09.2016
	 */
	public static void executeRhinoScript(HttpServletRequest request, UsuarioDTO usuario, SmartReportDTO smartReportDTO, List<Object[]> listValues, List<String> listAlias, StringBuilder htmlResult) {
		Context cx = Context.enter();
		Scriptable scope = cx.initStandardObjects();

		String sourceName = "UtilSmartReport_Script";

		scope.put("usuario", scope, usuario);
		scope.put("httpRequest", scope, request);

		scope.put("utilStrings", scope, new UtilStrings());
		scope.put("hashMapUtil", scope, new HashMapUtil());
		HashMap<String, Object> parametrosInformados = getParametrosInformados(request);

		if (parametrosInformados.isEmpty()) {
			for (SmartReportParameterDTO parametro : getParametros(smartReportDTO.getParametros())) {
				parametrosInformados.put("PARAM." + parametro.getName(), parametro.getDefaultValue());
			}
		}
		scope.put("hashParametros", scope, parametrosInformados);

		scope.put("jdbcEngine", scope, new JdbcEngine(Constantes.getValue("DATABASE_ALIAS"), null));
		scope.put("SGBD_PRINCIPAL", scope, CITCorporeUtil.SGBD_PRINCIPAL);

		scope.put("QUERY_RESULT", scope, listValues);
		scope.put("RETURN_COLUMNS", scope, listAlias);
		scope.put("HTML_RESULT", scope, htmlResult);

		cx.evaluateString(scope, smartReportDTO.getScript(), sourceName, 1, null);
	}

	/**
	 * @param document
	 * @param request
	 * @param smartReportDTO
	 * @param listValues
	 *
	 * @author douglas.japiassu
	 * @since 20.09.2016
	 */
	private static void processChartReports(DocumentHTML document, HttpServletRequest request, SmartReportDTO smartReportDTO, List<Object[]> listValues) {
		try {
			if (listValues != null && !listValues.isEmpty()) {
				JSONArray labels = new JSONArray();

				Map<Object, List<Object>> mapDatasets = new HashMap<>();

				for (Object[] object : listValues) {
					for (int i = 0; i < object.length; i++) {
						if (i == 0) {
							labels.put(object[i]);
						} else {
							if (object[i] instanceof Integer || object[i] instanceof Double || object[i] instanceof Long) {
								if (mapDatasets.get(i) == null) {
									mapDatasets.put(i, new ArrayList<>());
								}

								mapDatasets.get(i).add(object[i]);
							} else {
								throw new LogicException(UtilI18N.internacionaliza(request, "construtorconsultas.resultadoNaoPermiteGrafico"));
							}
						}
					}
				}

				JSONArray datasets = new JSONArray();
				if (mapDatasets != null && !mapDatasets.isEmpty()) {
					for (Object key : mapDatasets.keySet()) {
						datasets.put(new JSONObject().put("label", key).put("data", mapDatasets.get(key)));
					}
				}

				JSONObject data = new JSONObject();
				data.put("labels", labels).put("datasets", datasets);

				JSONObject options = new JSONObject();
				if (ReportTypes.BAR_CHART.getId().equals(smartReportDTO.getTipoRelatorio())) {
					options.put("scales", new JSONObject().put("yAxes", new JSONArray().put(new JSONObject().put("ticks", new JSONObject().put("beginAtZero", true)))));
				}
				options.put("responsive", true);
				options.put("legend", new JSONObject().put("display", ReportTypes.LINE_CHART.getId().equals(smartReportDTO.getTipoRelatorio()) || ReportTypes.BAR_CHART.getId().equals(smartReportDTO.getTipoRelatorio()) ? false : true));
				options.put("title", new JSONObject().put("display", true).put("text", smartReportDTO.getIdentificacao()));

				document.executeScript(
						"UtilSmartReport.buildSample(" + smartReportDTO.getIdSmartReport() + ", '" + smartReportDTO.getTipoRelatorio() + "', '" + StringEscapeUtils.escapeJavaScript(data.toString()) + "', '"
								+ StringEscapeUtils.escapeJavaScript(options.toString()) + "');");
			} else {
				// document.executeScript("notificacaoErro('construtorconsultas.naoFoiPossivelConstruirRelatorio', 'MSG04');");
				document.executeScript("UtilSmartReport.reportError(" + smartReportDTO.getIdSmartReport() + ", 'MSG04')");
			}
		} catch (Exception e) {
			String erro = e.getMessage() != null && !e.getMessage().isEmpty() ? StringEscapeUtils.escapeJavaScript(e.getMessage()) : "";
			// document.executeScript("notificacaoErro('construtorconsultas.naoFoiPossivelConstruirRelatorio', '" + erro + "');");
			document.executeScript("UtilSmartReport.reportError(" + smartReportDTO.getIdSmartReport() + ", '" + erro + "')");
		}
	}

	/**
	 *
	 * @param document
	 * @param request
	 * @param usuario
	 * @param smartReportDTO
	 * @param listValues
	 *
	 * @author douglas.japiassu
	 * @since 20.09.2016
	 */
	private static void processTemplateReport(DocumentHTML document, HttpServletRequest request, UsuarioDTO usuario, SmartReportDTO smartReportDTO, List<Object[]> listValues) {
		try {
			String path = "templateFreeMarker/" + usuario.getIdUsuario() + "/" + smartReportDTO.getIdentificacao();
			String fullPath = CITCorporeUtil.CAMINHO_REAL_APP + path;

			File f = new File(CITCorporeUtil.CAMINHO_REAL_APP + "/templateFreeMarker/" + usuario.getIdUsuario());
			if (!f.exists()) {
				f.mkdirs();
			}

			UtilTratamentoArquivos.geraFileTxtFromString(fullPath + ".ftl", CitAjaxUtil.encodeHTML(smartReportDTO.getTemplate()));

			Configuration cfg = new Configuration();
			cfg.setDirectoryForTemplateLoading(new File(CITCorporeUtil.CAMINHO_REAL_APP));
			Template temp = cfg.getTemplate(path + ".ftl");

			Map<String, Object> d = new HashMap<String, Object>();
			d.put("QUERY_RESULT", listValues);

			Writer file = new FileWriter(new File(fullPath + ".html"));
			temp.process(d, file);

			file.flush();
			file.close();

			String dados = UtilTratamentoArquivos.getStringTextFromFileTxtSemErroDeCharset(fullPath + ".html");

			document.executeScript("UtilSmartReport.buildSample(" + smartReportDTO.getIdSmartReport() + ", '" + smartReportDTO.getTipoRelatorio() + "', '" + StringEscapeUtils.escapeJavaScript(dados) + "');");

			Files.delete(Paths.get(fullPath + ".ftl"));
			Files.delete(Paths.get(fullPath + ".html"));
		} catch (Exception e) {
			document.executeScript("notificacaoErro('', '" + UtilI18N.internacionaliza(request, "construtorconsultas.naoFoiPossivelConstruirRelatorio") + "."
					+ (e.getMessage() != null && !e.getMessage().isEmpty() ? "<br/>" + UtilI18N.internacionaliza(request, "citcorpore.comum.erro") + ": " + StringEscapeUtils.escapeJavaScript(e.getMessage()) : "") + "');");
		}
	}

	/**
	 *
	 * @param document
	 * @param smartReportDTO
	 * @param listValues
	 * @param listAlias
	 * @throws JSONException
	 *
	 * @author douglas.japiassu
	 * @since 20.09.2016
	 */
	private static void processCrossReferencesReport(DocumentHTML document, SmartReportDTO smartReportDTO, List<Object[]> listValues, List<String> listAlias) throws JSONException {
		JSONArray jaDados = new JSONArray();

		for (Object[] obj : listValues) {
			JSONObject joDados = new JSONObject();
			for (int i = 0; i < listAlias.size(); i++) {
				joDados.put(listAlias.get(i), obj[i]);
			}
			jaDados.put(joDados);
		}

		document.executeScript("UtilSmartReport.buildSample(" + smartReportDTO.getIdSmartReport() + ", '" + smartReportDTO.getTipoRelatorio() + "', '" + StringEscapeUtils.escapeJavaScript(jaDados.toString()) + "');");
	}

	/**
	 *
	 * @param document
	 * @param smartReportDTO
	 * @param listValues
	 * @param listAlias
	 * @throws JSONException
	 *
	 * @author douglas.japiassu
	 * @since 20.09.2016
	 */
	private static void processDataTableReport(DocumentHTML document, SmartReportDTO smartReportDTO, List<Object[]> listValues, List<String> listAlias) throws JSONException {
		JSONObject dados = new JSONObject();
		JSONArray jaResult = new JSONArray();
		JSONArray jaColumns = new JSONArray();

		for (String alias : listAlias) {
			jaColumns.put(new JSONObject().put("title", alias));
		}
		for (Object[] obj : listValues) {
			jaResult.put(obj);
		}

		dados.put("columns", jaColumns);
		dados.put("result", jaResult);

		document.executeScript("UtilSmartReport.buildSample(" + smartReportDTO.getIdSmartReport() + ", '" + smartReportDTO.getTipoRelatorio() + "', '" + StringEscapeUtils.escapeJavaScript(dados.toString()) + "');");
	}

	private static SmartReportService getSmartReportService() throws ServiceException, Exception {
		if (smartReportService == null) {
			smartReportService = (SmartReportService) ServiceLocator.getInstance().getService(SmartReportService.class, null);
		}

		return smartReportService;
	}

}
