package br.com.citframework.util;

import java.sql.Timestamp;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

import br.com.centralit.citcorpore.bean.CalculoJornadaDTO;
import br.com.centralit.citcorpore.bean.CalendarioDTO;
import br.com.centralit.citcorpore.bean.JornadaTrabalhoDTO;
import br.com.centralit.citcorpore.integracao.CalendarioDao;
import br.com.centralit.citcorpore.integracao.JornadaTrabalhoDao;
import br.com.centralit.citcorpore.negocio.CalendarioServiceEjb;
import br.com.centralit.citcorpore.util.CITCorporeUtil;
import br.com.centralit.citcorpore.util.Enumerados.TipoDate;
import br.com.citframework.excecao.PersistenceException;
import br.com.citframework.integracao.TransactionControler;

/**
 * Classe de testes para validacao do comportamento de
 * {@link CalendarioServiceEjb}
 * 
 * Os dados da classe levam em consideracao os seguintes Calendarios e suas
 * respectivas JornadaTrabalho
 * 
 * idCalendario: Calendario de seg a sex Jornada de Trabalho das 08:00 as 20:00
 * 
 * idCalendario: Calendario de seg a sab Jornada de Trabalho das 08:00 - 12:00 e
 * das 14:00 a 18:00 - Seg a Sex Jornada de Trabalho das 08:00 - 12:00 - Sab
 *
 * @author gilberto.nery - <a href=
 *         "mailto:gilberto.nery@centrait.com.br">gilberto.nery@centrait.com.br</a>
 * @since 2016-09-25
 */
@RunWith(value = MockitoJUnitRunner.class)
@Ignore
public class CalendarioServiceEjbTest {

	private CalendarioServiceEjb calendario = new CalendarioServiceEjb();

	private static Integer idCalendarioDiaDeSemanaSemIntervalo;
	private static Integer idJornadaDiaDeSemanaSemIntervalo;

	private static Integer idCalendario24Horas;
	private static Integer idJornada24Horas;

	private static Integer idCalendarioCriadoComIntervalo;
	private static Integer idJornadaCriadaDiaDeSemana;
	private static Integer idJornadaCriadaFimDeSemana;

	private static Integer idCalendarioCriadoComIntervaloSemFinalDeSemana;
	
	private static Integer idCalendario3Turnos;
	private static Integer idJornada3Turnos;

	private static Integer idJornadaTurnosHorasCorretoPassandoDia;
	private static Integer idCalendarioTurnosHorasCorretoPassandoDia;

	@Before
	public void setarConfiguracoesJDBC() {
		CITCorporeUtil.SGBD_PRINCIPAL = SQLConfig.POSTGRESQL;
		Constantes.setClasseTeste();
	}

	@BeforeClass
	public static void inserirCalendarioEJornadaComIntervalo() {

		CITCorporeUtil.SGBD_PRINCIPAL = SQLConfig.POSTGRESQL;
		Constantes.setClasseTeste();

		new CalendarioServiceEjbTest().preencherMassaDeTeste();
	}

	private void preencherMassaDeTeste() {
		
		try {

			CalendarioDao calendarioDao = new CalendarioDao();
			JornadaTrabalhoDao jornadaDao = new JornadaTrabalhoDao();
			
			// Calendario para testar jornadas de trabalho sem intervalo
			// Seg a sexta das 08:00 a 20:00
			// Jornada dia de semana
			JornadaTrabalhoDTO jornadaSemIntervalo = new JornadaTrabalhoDTO();
			jornadaSemIntervalo.setDescricao("Jornada dia de semana sem intervalo");
			jornadaSemIntervalo.setInicio1("08:00");
			jornadaSemIntervalo.setTermino1("20:00");
			jornadaSemIntervalo.setCargaHoraria("12:00");
			jornadaSemIntervalo.setTipoHorario(1);
			
			jornadaSemIntervalo = (JornadaTrabalhoDTO) jornadaDao.create(jornadaSemIntervalo);
			idJornadaDiaDeSemanaSemIntervalo = jornadaSemIntervalo.getIdJornada();
	
			CalendarioDTO calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo = new CalendarioDTO();
			calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo.setConsideraFeriados("S");
			calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo.setDescricao("Teste unitario de integracao sem intervalo");
			calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo.setIdJornadaSeg(idJornadaDiaDeSemanaSemIntervalo);
			calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo.setIdJornadaTer(idJornadaDiaDeSemanaSemIntervalo);
			calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo.setIdJornadaQua(idJornadaDiaDeSemanaSemIntervalo);
			calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo.setIdJornadaQui(idJornadaDiaDeSemanaSemIntervalo);
			calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo.setIdJornadaSex(idJornadaDiaDeSemanaSemIntervalo);
			calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo = (CalendarioDTO) calendarioDao.create(calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo);
			idCalendarioDiaDeSemanaSemIntervalo = calendarioDeSegASexComJornadaDe60hSemanaisSemIntervalo.getIdCalendario();
						
			//	----------------------------------------------------------------------------------- //
			
			// Calendario para testar jornadas de trabalho com intervalo
			// Seg a sexta das 08:00 a 12:00 e das 14:00 as 18:00
			// Sab das 08:00 as 12:00
			// Jornada dia de semana
			JornadaTrabalhoDTO jornada = new JornadaTrabalhoDTO();
			jornada.setDescricao("Jornada dia de semana");
			jornada.setInicio1("08:00");
			jornada.setTermino1("12:00");
			jornada.setInicio2("14:00");
			jornada.setTermino2("18:00");
			jornada.setCargaHoraria("08:00");
			jornada.setTipoHorario(1);
			
			jornada = (JornadaTrabalhoDTO) jornadaDao.create(jornada);
			idJornadaCriadaDiaDeSemana = jornada.getIdJornada();
	
			// Jornada fim de semana
			JornadaTrabalhoDTO jornada2 = new JornadaTrabalhoDTO();
			jornada2.setDescricao("Jornada fim de semana");
			jornada2.setInicio1("08:00");
			jornada2.setTermino1("12:00");
			jornada2.setCargaHoraria("04:00");
			jornada2.setTipoHorario(1);
			
			jornada2 = (JornadaTrabalhoDTO) jornadaDao.create(jornada2);
			idJornadaCriadaFimDeSemana = jornada2.getIdJornada();
			
			CalendarioDTO calendarioDeSegASabComJornadaDe44hSemanais = new CalendarioDTO();
			calendarioDeSegASabComJornadaDe44hSemanais.setConsideraFeriados("S");
			calendarioDeSegASabComJornadaDe44hSemanais.setDescricao("Teste unitario de integracao com intervalo");
			calendarioDeSegASabComJornadaDe44hSemanais.setIdJornadaSeg(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASabComJornadaDe44hSemanais.setIdJornadaTer(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASabComJornadaDe44hSemanais.setIdJornadaQua(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASabComJornadaDe44hSemanais.setIdJornadaQui(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASabComJornadaDe44hSemanais.setIdJornadaSex(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASabComJornadaDe44hSemanais.setIdJornadaSab(idJornadaCriadaFimDeSemana);
			calendarioDeSegASabComJornadaDe44hSemanais = (CalendarioDTO) calendarioDao.create(calendarioDeSegASabComJornadaDe44hSemanais);
			idCalendarioCriadoComIntervalo = calendarioDeSegASabComJornadaDe44hSemanais.getIdCalendario();

			//	----------------------------------------------------------------------------------- //

			
			// Calendario para testar jornadas de trabalho com intervalo de segunda a sabado
			// Seg a sexta das 08:00 a 12:00 e das 14:00 as 18:00
			// Jornada dia de semana
			
			CalendarioDTO calendarioDeSegASex40hSemanais = new CalendarioDTO();
			calendarioDeSegASex40hSemanais.setConsideraFeriados("S");
			calendarioDeSegASex40hSemanais.setDescricao("Teste unitario de integracao com intervalo de seg a sab");
			calendarioDeSegASex40hSemanais.setIdJornadaSeg(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASex40hSemanais.setIdJornadaTer(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASex40hSemanais.setIdJornadaQua(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASex40hSemanais.setIdJornadaQui(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASex40hSemanais.setIdJornadaSex(idJornadaCriadaDiaDeSemana);
			calendarioDeSegASex40hSemanais = (CalendarioDTO) calendarioDao.create(calendarioDeSegASex40hSemanais);
			idCalendarioCriadoComIntervaloSemFinalDeSemana = calendarioDeSegASex40hSemanais.getIdCalendario();

			//	----------------------------------------------------------------------------------- //

						
						
			// Calendario para testar jornadas de trabalho com 24 horas
			// Seg a dom das 00:00 a 24:00			
			// Jornada 24horas
			JornadaTrabalhoDTO jornada24Horas = new JornadaTrabalhoDTO();
			jornada24Horas.setDescricao("Jornada sem intervalo 24 horas");
			jornada24Horas.setInicio1("00:00");
			jornada24Horas.setTermino1("24:00");
			jornada24Horas.setCargaHoraria("24:00");
			jornada24Horas.setTipoHorario(1);
			
			jornada24Horas = (JornadaTrabalhoDTO) jornadaDao.create(jornada24Horas);
			idJornada24Horas = jornada24Horas.getIdJornada();
			
			CalendarioDTO calendario24hSemanais = new CalendarioDTO();
			calendario24hSemanais.setConsideraFeriados("N");
			calendario24hSemanais.setDescricao("Teste unitario de integracao sem intervalo 24 horas");
			calendario24hSemanais.setIdJornadaSeg(idJornada24Horas);
			calendario24hSemanais.setIdJornadaTer(idJornada24Horas);
			calendario24hSemanais.setIdJornadaQua(idJornada24Horas);
			calendario24hSemanais.setIdJornadaQui(idJornada24Horas);
			calendario24hSemanais.setIdJornadaSex(idJornada24Horas);
			calendario24hSemanais.setIdJornadaSab(idJornada24Horas);
			calendario24hSemanais.setIdJornadaDom(idJornada24Horas);

			calendario24hSemanais = (CalendarioDTO) calendarioDao.create(calendario24hSemanais);
			idCalendario24Horas = calendario24hSemanais.getIdCalendario();
						
			//	----------------------------------------------------------------------------------- //

			// Calendario para testar jornadas de trabalho com 3 turnos
			// Jornada 3 turnos
			JornadaTrabalhoDTO jornada3Turnos = new JornadaTrabalhoDTO();
			jornada3Turnos.setDescricao("Jornada com 3 turnos");
			jornada3Turnos.setInicio1("09:00");
			jornada3Turnos.setTermino1("13:00");
			jornada3Turnos.setInicio2("14:00");
			jornada3Turnos.setTermino2("18:00");
			jornada3Turnos.setInicio3("18:15");
			jornada3Turnos.setTermino3("21:45");
			jornada3Turnos.setCargaHoraria("11:30");
			jornada3Turnos.setTipoHorario(1);
			
			jornada3Turnos = (JornadaTrabalhoDTO) jornadaDao.create(jornada3Turnos);
			idJornada3Turnos = jornada3Turnos.getIdJornada();
			
			CalendarioDTO calendario3Turnos = new CalendarioDTO();
			calendario3Turnos.setConsideraFeriados("N");
			calendario3Turnos.setDescricao("Teste unitario de integracao sem intervalo 24 horas");
			calendario3Turnos.setIdJornadaSeg(idJornada3Turnos);
			calendario3Turnos.setIdJornadaTer(idJornada3Turnos);
			calendario3Turnos.setIdJornadaQua(idJornada3Turnos);
			calendario3Turnos.setIdJornadaQui(idJornada3Turnos);
			calendario3Turnos.setIdJornadaSex(idJornada3Turnos);

			calendario3Turnos = (CalendarioDTO) calendarioDao.create(calendario3Turnos);
			idCalendario3Turnos = calendario3Turnos.getIdCalendario();

			//	----------------------------------------------------------------------------------- //

			// Calendario para testar jornadas de trabalho com exemplo na India
			// Seg a dom das 00:00 a 06:00 e das 18:00 as 24:00			
			// Jornada 24horas
			JornadaTrabalhoDTO jornadaIndia = new JornadaTrabalhoDTO();
			jornadaIndia.setDescricao("Jornada India");
			jornadaIndia.setInicio1("00:00");
			jornadaIndia.setTermino1("06:00");
			jornadaIndia.setInicio2("18:00");
			jornadaIndia.setTermino2("24:00");
			jornadaIndia.setCargaHoraria("12:00");
			jornadaIndia.setTipoHorario(1);
			
			jornadaIndia = (JornadaTrabalhoDTO) jornadaDao.create(jornadaIndia);
			idJornadaTurnosHorasCorretoPassandoDia = jornadaIndia.getIdJornada();
			
			CalendarioDTO calendarioIndia = new CalendarioDTO();
			calendarioIndia.setConsideraFeriados("N");
			calendarioIndia.setDescricao("Calendario India");
			calendarioIndia.setIdJornadaSeg(idJornadaTurnosHorasCorretoPassandoDia);
			calendarioIndia.setIdJornadaTer(idJornadaTurnosHorasCorretoPassandoDia);
			calendarioIndia.setIdJornadaQua(idJornadaTurnosHorasCorretoPassandoDia);
			calendarioIndia.setIdJornadaQui(idJornadaTurnosHorasCorretoPassandoDia);
			calendarioIndia.setIdJornadaSex(idJornadaTurnosHorasCorretoPassandoDia);
			calendarioIndia.setIdJornadaSab(idJornadaTurnosHorasCorretoPassandoDia);
			calendarioIndia.setIdJornadaDom(idJornadaTurnosHorasCorretoPassandoDia);

			calendarioIndia = (CalendarioDTO) calendarioDao.create(calendarioIndia);
			idCalendarioTurnosHorasCorretoPassandoDia = calendarioIndia.getIdCalendario();

			//	----------------------------------------------------------------------------------- //
			
		} catch (PersistenceException e) {
			e.printStackTrace();
		}
	}

	@Mock
	private TransactionControler tc;

	@Test(expected = Exception.class)
	public void testIdDoCalendarioNaoInformado() throws Exception {
		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		dataHoraInicial = UtilDatas.getDataHoraAtual();
		dataHoraFinal = UtilDatas.getDataHoraAtual();
		calendario.validaObjetoCalculoJornada(new CalculoJornadaDTO(null, dataHoraInicial, dataHoraFinal));
	}

	@Test(expected = Exception.class)
	public void testDataIncialNaoInformada() throws Exception {
		Timestamp dataHoraFinal = UtilDatas.getDataHoraAtual();
		calendario.validaObjetoCalculoJornada(
				new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, null, dataHoraFinal));
	}

	@Test(expected = Exception.class)
	public void testDataFinalNaoInformada() throws Exception {
		Timestamp dataHoraInicial = UtilDatas.getDataHoraAtual();
		calendario.validaObjetoCalculoJornada(
				new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial, null));
	}

	@Test(expected = Exception.class)
	public void testDataInicalMaiorQueAFinal() throws Exception {
		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		dataHoraInicial = UtilDatas.somaSegundos(UtilDatas.getDataHoraAtual(), 1);
		dataHoraFinal = UtilDatas.getDataHoraAtual();
		calendario.validaObjetoCalculoJornada(
				new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial, dataHoraFinal));
	}

	@Test
	public void testCalculoForaDaJornadaDeTrabalho() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			
			// Data Hora no fim de semana - domingo
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "25/09/2016 06:00:00.000", "PT");
			// 2h depois da hora inicial
			dataHoraFinal = UtilDatas.somaSegundos(dataHoraInicial, 7200);
			
			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoTotalEmSegundos().equals(0d));
			Assert.assertTrue(calculo.getTempoDecorridoTotalEmSegundos().equals(calculo2.getTempoDecorridoTotalEmSegundos()));
			
			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);
	
			Assert.assertTrue(calculo3.getTempoDecorridoTotalEmSegundos().equals(0d));
			Assert.assertTrue(calculo3.getTempoDecorridoTotalEmSegundos().equals(calculo4.getTempoDecorridoTotalEmSegundos()));

			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "10/10/2016 21:33:42.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "11/10/2016 05:33:42.000", "PT");
			
			calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoTotalEmSegundos().equals(0d));
			
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "10/10/2016 21:33:42.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "11/10/2016 19:00:00.000", "PT");
			
			calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoTotalEmSegundos().equals(39600d));
			
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "10/10/2016 21:33:42.000", "PT");
			dataHoraFinal = UtilDatas.somaSegundos(dataHoraInicial, 35);
			
			calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoTotalEmSegundos().equals(0d));
			
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "10/10/2016 21:33:42.000", "PT");
			dataHoraFinal = UtilDatas.somaSegundos(dataHoraInicial, 7200);
			
			calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoTotalEmSegundos().equals(0d));
			
		} catch (Exception e) {
			e.printStackTrace();
			Assert.assertTrue("Algo deu errado", false);
		}
	}

	@Test
	public void testCalculoIniciaDentroDaJornadaDeTrabalhoEEncerraFora() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			// Data Hora no dia de semana - sexta
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"23/09/2016 16:45:00.000", "PT");
			// 36h depois
			dataHoraFinal = UtilDatas.somaSegundos(dataHoraInicial, 129600);

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
		
			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(3));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(15));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
			
			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);
		
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(5));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(15));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
			
		} catch (Exception e) {
			e.printStackTrace();
			Assert.assertTrue("Algo deu errado", false);
		}
	}

	@Test
	public void testCalculoIniciaEEncerraDentroDaJornadaDeTrabalho() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			// Data Hora no dia de semana - sexta
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"23/09/2016 11:03:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"23/09/2016 17:35:00.000", "PT");
			
			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);

			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(6));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(32));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
			
			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(4));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(32));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
			Assert.assertTrue("Algo deu errado", false);
		}
	}

	@Test
	public void testCalculoIniciaEEncerraDentroDaJornadaDeTrabalhoComIntervaloDe3Dias() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			// Data Hora no dia de semana - terca
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"20/09/2016 09:30:00.000", "PT");
			// Data Hora no dia de semana - sexta
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"23/09/2016 17:50:00.000", "PT");
			
			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			
			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);

			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(44));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(20));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
			
			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(30));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(20));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	@Test
	public void testCalculoIniciaForaDaJornadaDeTrabalhoEEncerraDentro() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			// Data Hora no dia de semana - SEXTA
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"23/09/2016 22:10:22.000", "PT");
			// Data Hora no dia de semana - terca
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"04/10/2016 17:50:21.000", "PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(81));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(50));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));

			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);
		
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(63));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(50));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	@Test
	public void testCalculoIniciaAntesDaJornadaDeTrabalhoEEncerraDentroNoMesmoDia() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"23/09/2016 07:27:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"23/09/2016 15:13:00.000", "PT");
			
			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(7));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(13));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));

			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(5));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(13));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	@Test
	public void testCalculoIniciaEEncerraForaDaJornadaDeTrabalhoComIntervaloDe15Dias() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			// Data Hora no fim de semana - domingo
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "25/09/2016 09:30:00","PT");
			// Data Hora em dia de semana mas fora do expediente de trabalho
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "10/10/2016 21:12:00","PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(132));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(0));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
			
			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(96));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(0));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoIniciaDentroDaJornadaDeTrabalhoEEncerraNoIntervaloDo2Dia() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "26/09/2016 09:30:00","PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "28/09/2016 13:00:00","PT");
			
			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);

			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(18));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(30));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoIniciaDentroDoIntervaloDaJornadaEEncerraNoIntervaloDo2Dia() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "26/09/2016 12:30:00","PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "28/09/2016 13:00:00","PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);

			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(16));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(0));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoIniciaForaDoIntervaloDaJornadaEEncerraDentroDaJornada() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "26/09/2016 07:00:00","PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "28/09/2016 17:26:00","PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);

			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(23));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(26));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculo24HorasDeJornadaDeTrabalho() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "27/09/2016 09:16:00","PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "29/09/2016 17:00:00","PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendario24Horas, dataHoraInicial, dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendario24Horas, dataHoraInicial, dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);

			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(55));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(44));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
		
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoCom3JornadasDeTrabalho() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "27/09/2016 15:36:00","PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "03/10/2016 14:58:00","PT");

			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendario3Turnos, dataHoraInicial, dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendario3Turnos, dataHoraInicial, dataHoraFinal);

			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(45));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(22));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoCom3JornadasDeTrabalhoPassandoUmDia() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "29/09/2016 08:43:00","PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "04/10/2016 20:00:00","PT");

			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendario3Turnos, dataHoraInicial, dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendario3Turnos, dataHoraInicial, dataHoraFinal);

			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(44));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(15));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoCom3JornadasDeTrabalhoAbrindoDentroEFechandoFora() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "28/09/2016 14:50:00","PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "30/09/2016 11:01:00","PT");

			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendario3Turnos, dataHoraInicial, dataHoraFinal);
			CalculoJornadaDTO calculo4 = new CalculoJornadaDTO(idCalendario3Turnos, dataHoraInicial, dataHoraFinal);

			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);
			calculo4 = calendario.calculaPrazoDecorrido(calculo4, dataHoraFinal, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(20));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(11));
	
			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(calculo4.getTempoDecorridoHH()));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(calculo4.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoMesmoDiaAlgunsMinutosDepois() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"29/09/2016 08:35:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"29/09/2016 09:06:00.000", "PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);
		
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(0));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(31));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoJornada24HorasDaIndia() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "29/09/2016 20:35:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS, "30/09/2016 04:06:00.000", "PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioTurnosHorasCorretoPassandoDia, dataHoraInicial, dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
		
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(7));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(31));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Test
	public void testCalculoAbreEFechaNaSegundaJornadaDoMesmoDia() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"29/09/2016 14:35:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"29/09/2016 17:06:00.000", "PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);
		
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(2));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(31));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}
	
	@Test
	public void testCalculoAbreEFechaNaSegundaJornadaPassandoUmDia() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"29/09/2016 14:35:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"30/09/2016 17:06:00.000", "PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioCriadoComIntervaloSemFinalDeSemana, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendarioCriadoComIntervaloSemFinalDeSemana, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaPrazoDecorrido(calculo2, dataHoraFinal, tc);
		
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(10));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(31));
	
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(calculo2.getTempoDecorridoHH()));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(calculo2.getTempoDecorridoMM()));
		} catch (Exception e) {
			e.printStackTrace();
		}

	}
	
	@Test
	public void testCalculoTurnosDe24HorasPassando20Dias() {

		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"09/09/2016 14:35:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"29/09/2016 17:06:00.000", "PT");

			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioTurnosHorasCorretoPassandoDia, dataHoraInicial,dataHoraFinal);
		
			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(240));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(0));
		
		} catch (Exception e) {
			e.printStackTrace();
		}
		
	}
	
	@Test
	public void testCalculoTurnosEmCimaDaJornada() {
		
		Timestamp dataHoraInicial; 
		Timestamp dataHoraFinal;
		
		try {
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"21/09/2016 14:00:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"29/09/2016 18:00:00.000", "PT");
			
			CalculoJornadaDTO calculo = new CalculoJornadaDTO(idCalendarioCriadoComIntervalo, dataHoraInicial,dataHoraFinal);
			CalculoJornadaDTO calculo2 = new CalculoJornadaDTO(idCalendario3Turnos, dataHoraInicial,dataHoraFinal);

			calculo = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo, tc);
			calculo2 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo2, tc);
			
			Assert.assertTrue(calculo.getTempoDecorridoHH().equals(56));
			Assert.assertTrue(calculo.getTempoDecorridoMM().equals(0));
	
			Assert.assertTrue(calculo2.getTempoDecorridoHH().equals(73));
			Assert.assertTrue(calculo2.getTempoDecorridoMM().equals(00));
			
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"30/09/2016 17:22:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"01/10/2016 15:00:00.000", "PT");
			
			CalculoJornadaDTO calculo3 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial, dataHoraFinal);
		
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(2));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(38));
			
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"30/09/2016 17:22:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"03/10/2016 08:00:00.000", "PT");
			
			calculo3 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial, dataHoraFinal);
		
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(2));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(38));
			
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"30/09/2016 20:00:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"03/10/2016 08:00:00.000", "PT");
			
			calculo3 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial, dataHoraFinal);
		
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(0));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(0));
			
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"30/09/2016 20:00:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"03/10/2016 20:00:00.000", "PT");
			
			calculo3 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial, dataHoraFinal);
		
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(12));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(0));
			
			dataHoraInicial = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"30/09/2016 20:00:00.000", "PT");
			dataHoraFinal = UtilDatas.convertStringToTimestamp(TipoDate.TIMESTAMP_WITH_SECONDS,"04/10/2016 08:00:00.000", "PT");
			
			calculo3 = new CalculoJornadaDTO(idCalendarioDiaDeSemanaSemIntervalo, dataHoraInicial, dataHoraFinal);
		
			calculo3 = calendario.calculaTempoDecorridoPelaJornadaDeTrabalho(calculo3, tc);

			Assert.assertTrue(calculo3.getTempoDecorridoHH().equals(12));
			Assert.assertTrue(calculo3.getTempoDecorridoMM().equals(0));
			
		} catch (Exception e) {
			e.printStackTrace();
		}

	}
	
	@AfterClass
	public static void tearDown(){
		new CalendarioServiceEjbTest().limparConfiguracoesDeTeste();
	}
	
	/**
	 * Limpa a massa de dados cadastrados para realizar o teste
	 */
	public void limparConfiguracoesDeTeste() {

		try {
			CalendarioDao calendarioDao = new CalendarioDao();
			JornadaTrabalhoDao jornadaDao = new JornadaTrabalhoDao();
			
			CalendarioDTO calendario = new CalendarioDTO();
			calendario.setIdCalendario(idCalendarioDiaDeSemanaSemIntervalo);
			calendarioDao.delete(calendario);
			
			calendario = new CalendarioDTO();
			calendario.setIdCalendario(idCalendario24Horas);
			calendarioDao.delete(calendario);
			
			calendario = new CalendarioDTO();
			calendario.setIdCalendario(idCalendario3Turnos);
			calendarioDao.delete(calendario);
			
			calendario = new CalendarioDTO();
			calendario.setIdCalendario(idCalendarioCriadoComIntervalo);
			calendarioDao.delete(calendario);
			
			calendario = new CalendarioDTO();
			calendario.setIdCalendario(idCalendarioCriadoComIntervaloSemFinalDeSemana);
			calendarioDao.delete(calendario);
			
			calendario = new CalendarioDTO();
			calendario.setIdCalendario(idCalendarioTurnosHorasCorretoPassandoDia);
			calendarioDao.delete(calendario);
			
			JornadaTrabalhoDTO jornadaASeremExcluidas = new JornadaTrabalhoDTO();
			jornadaASeremExcluidas.setIdJornada(idJornadaDiaDeSemanaSemIntervalo);
			jornadaDao.delete(jornadaASeremExcluidas);
			
			jornadaASeremExcluidas = new JornadaTrabalhoDTO();
			jornadaASeremExcluidas.setIdJornada(idJornada3Turnos);
			jornadaDao.delete(jornadaASeremExcluidas);
			
			jornadaASeremExcluidas = new JornadaTrabalhoDTO();
			jornadaASeremExcluidas.setIdJornada(idJornadaCriadaFimDeSemana);
			jornadaDao.delete(jornadaASeremExcluidas);
			
			jornadaASeremExcluidas = new JornadaTrabalhoDTO();
			jornadaASeremExcluidas.setIdJornada(idJornadaCriadaDiaDeSemana);
			jornadaDao.delete(jornadaASeremExcluidas);
			
			jornadaASeremExcluidas = new JornadaTrabalhoDTO();
			jornadaASeremExcluidas.setIdJornada(idJornada24Horas);
			jornadaDao.delete(jornadaASeremExcluidas);
			
			jornadaASeremExcluidas = new JornadaTrabalhoDTO();
			jornadaASeremExcluidas.setIdJornada(idJornadaTurnosHorasCorretoPassandoDia);
			jornadaDao.delete(jornadaASeremExcluidas);

		} catch (PersistenceException e) {
			e.printStackTrace();
		}
	}

}