/**********************************LICENCA*GPLv2********************************************************************
* Copyright [2011,2012,2013,2014,2015,2016] da CentralIT Tecnologia da Informao Ltda (www.centralit.com.br)      *
*                                                                                                                  *
* Este arquivo  parte do programa/software: Citsmart (www.citsmart.com.br)                                        *
*                                                                                                                  *
* O Citsmart  um software livre; voc pode redistribui-lo e/ou modific-lo dentro dos termos da Licena           *
* Pblica Geral GNU como publicada pela Fundao do Software Livre (FSF); na verso 2 da Licena.                  *
*                                                                                                                  *
* Este programa/software  distribudo na esperana que possa ser til, mas SEM NENHUMA GARANTIA; sem uma          *
* garantia implcita de ADEQUAO a qualquer MERCADO ou APLICAO EM PARTICULAR. Veja a Licena Pblica Geral      *
* GNU/GPL em portugus para maiores detalhes.                                                                      *
*                                                                                                                  *
* Voc deve ter recebido uma cpia da Licena Pblica Geral GNU, sob o ttulo 'LICENCA.txt', junto com este        *
* programa/software, se no, acesse o Portal do Software Pblico Brasileiro no endereo www.softwarepublico.gov.br *
* ou escreva para a Fundao do Software Livre (FSF) Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301,USA  *
********************************************************************************************************************/
package br.com.centralit.agents;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.snmp4j.CommunityTarget;
import org.snmp4j.PDU;
import org.snmp4j.Snmp;
import org.snmp4j.Target;
import org.snmp4j.TransportMapping;
import org.snmp4j.event.ResponseEvent;
import org.snmp4j.mp.SnmpConstants;
import org.snmp4j.smi.Address;
import org.snmp4j.smi.GenericAddress;
import org.snmp4j.smi.OID;
import org.snmp4j.smi.OctetString;
import org.snmp4j.smi.VariableBinding;
import org.snmp4j.transport.DefaultUdpTransportMapping;

import br.com.centralit.bean.HostContent;
import br.com.centralit.bean.HostItem;
import br.com.centralit.util.FileUtils;
import br.com.centralit.util.InventoryAttributes;
import br.com.centralit.util.RuntimeScript;
import br.com.centralit.util.ScriptRhinoJSExecute;
import br.com.centralit.util.UtilStrings;

public class SNMPManager implements AutoCloseable {

	private static final Logger LOGGER = Logger.getLogger(SNMPManager.class);

	private Snmp snmp = null;
	private String address = null;
	private String community = null;

	private SNMPManager(final String address, final String community) {
		this.address = address;
		this.community = community;
	}

	public static HostContent getHostContentBySNMP(final String ipHostName, final int port, final String communityParm) throws Exception {
		final HostContent host = new HostContent();
		String communitySearch = communityParm;
		Set<HostItem> hardware = SNMPManager.getSystemInfo(ipHostName, port, communitySearch);
		if (hardware.isEmpty()) {
			communitySearch = "public"; // Tenta tambem pela community public
			hardware = SNMPManager.getSystemInfo(ipHostName, port, communitySearch);
			if (hardware.isEmpty()) {
				return host;
			}
		}

		host.getHostContent().addAll(hardware);
		host.getHostContent().addAll(SNMPManager.getStoragesInfo(ipHostName, port, communitySearch));
		host.getHostContent().addAll(SNMPManager.getSoftwaresInfo(ipHostName, port, communitySearch));
		host.getHostContent().addAll(SNMPManager.getNetworksInfo(ipHostName, port, communitySearch));

		//Ler um diretorio do JBOSS - que pode ser ./scriptsSNMP
		final String rhino_diretorio_scripts = System.getProperty("rhino.scripts.directory");

		if (UtilStrings.isNotVazio(rhino_diretorio_scripts)) {
			//Ler todos os arquivos com extensao .script
			File folder = new File(rhino_diretorio_scripts);
			File[] listOfFiles = folder.listFiles();

			for (int i = 0; i < listOfFiles.length; i++) {

				if (listOfFiles[i].isFile()) {
					//Pra cada arquivo lido executa o codigo abaixo e joga o conteudo do arquivo para a variavel: "strScript"
					String sAux = rhino_diretorio_scripts + "/" + listOfFiles[i].getName();
					String strScript = FileUtils.readFile(sAux);

					if (strScript != null) {
						ScriptRhinoJSExecute scriptExecute = new ScriptRhinoJSExecute();
						RuntimeScript runtimeScript = new RuntimeScript();
						Context cx = Context.enter();
						Scriptable scope = cx.initStandardObjects();
						scope.put("ipHostName", scope, ipHostName);
						scope.put("host", scope, host);
						scope.put("out", scope, System.out);
						scope.put("RuntimeScript", scope, runtimeScript);
						scriptExecute.processScript(cx, scope, strScript, SNMPManager.class.getName() + "_" + sAux);

					}

				}

			}

		}

		return host;

	}

	private static Set<HostItem> getSystemInfo(final String ipHostName, final int port, final String communityParm) {
		final Set<HostItem> items = new HashSet<>();
		try (final SNMPManager client = new SNMPManager("udp:" + ipHostName + "/" + port, communityParm)) {
			client.start();
			final String sysDescr = client.getAsString(new OID(".1.3.6.1.2.1.1.1.0"));
			final String sysUpTime = client.getAsString(new OID(".1.3.6.1.2.1.1.3.0"));
			String sysName = client.getAsString(new OID(".1.3.6.1.2.1.1.5.0"));

			final String str = sysDescr;
			final int index = str.indexOf("Software:");
			String nomeHard = "";
			String nomeSO = "";
			if (index > -1) {
				final int indexHard = str.indexOf("Hardware:");
				if (indexHard > -1) {
					nomeHard = str.substring(indexHard + 9, index - 3);
				}
				nomeSO = str.substring(index + 9);
			}

			if (StringUtils.isNotBlank(nomeHard)) {
				nomeHard = sysDescr;
			}

			String workGroup = "";
			if (sysName.indexOf(".") > -1) {
				workGroup = sysName.substring(sysName.indexOf(".") + 1);
				sysName = sysName.substring(0, sysName.indexOf("."));
				sysName = sysName.replaceAll("\\.", "");
			}

			if (StringUtils.isBlank(nomeSO)) {
				if (nomeHard.indexOf("Linux") > -1) {
					nomeSO = "Linux";
				} else if (nomeHard.indexOf("Windows") > -1) {
					nomeSO = "Windows";
				}
			}

			if (StringUtils.isNotBlank(sysName)) {
				final HostItem item = new HostItem();
				item.setName(InventoryAttributes.HARDWARE);
				final Map<String, String> detail = new HashMap<>();
				detail.put(InventoryAttributes.NAME, sysName.toUpperCase().trim());
				detail.put(InventoryAttributes.PROCESSORT, nomeHard);
				detail.put(InventoryAttributes.OSNAME, nomeSO.trim());
				detail.put(InventoryAttributes.UPTIME, sysUpTime.trim());
				detail.put(InventoryAttributes.OSVERSION, nomeSO.trim());
				detail.put(InventoryAttributes.WORKGROUP, workGroup.trim());
				detail.put(InventoryAttributes.IPADDR, ipHostName.trim());
				item.setProperties(detail);
				items.add(item);
			}
		} catch (final IOException e) {
			LOGGER.debug(String.format("Citsmart Inventory > Erro ao recuperar informacao do host de IP/name %s: %s", ipHostName, e.getMessage()));
		}
		return items;
	}

	private static Set<HostItem> getStoragesInfo(final String ipHostName, final int port, final String communityParm) {
		final Set<HostItem> items = new HashSet<>();
		try (final SNMPManager client = new SNMPManager("udp:" + ipHostName + "/" + port, communityParm)) {
			client.start();
			for (int i = 1; i < 100; i++) {
				try {
					final String sysDiskType = client.getAsString(new OID(".1.3.6.1.2.1.25.2.3.1.2." + i));

					if (StringUtils.isBlank(sysDiskType)) {
						break;
					}

					final String sysDiskName = client.getAsString(new OID(".1.3.6.1.2.1.25.2.3.1.3." + i));

					if (StringUtils.isNotBlank(sysDiskName) && !sysDiskName.trim().equalsIgnoreCase("Null")) {
						final String alocUnitsStr = client.getAsString(new OID(".1.3.6.1.2.1.25.2.3.1.4." + i));
						final long alocUnits = parseLong(alocUnitsStr);

						final String sizeStr = client.getAsString(new OID(".1.3.6.1.2.1.25.2.3.1.5." + i));
						final long size = parseLong(sizeStr) * alocUnits;

						final String usedStr = client.getAsString(new OID(".1.3.6.1.2.1.25.2.3.1.6." + i));
						final long used = parseLong(usedStr) * alocUnits;

						final long free = size - used;

						if (StringUtils.isNotBlank(sysDiskName) && !sysDiskName.trim().equalsIgnoreCase("noSuchInstance") && !sysDiskName.trim().equalsIgnoreCase("noSuchObject")) {
							final HostItem item = new HostItem();
							item.setName(InventoryAttributes.STORAGES);
							final Map<String, String> detail = new HashMap<>();
							detail.put(InventoryAttributes.DESCRIPTION, sysDiskName);
							detail.put(InventoryAttributes.NAME, sysDiskName);
							detail.put(InventoryAttributes.DISKSIZE, "" + size);
							detail.put(InventoryAttributes.FREE, "" + free);
							item.setProperties(detail);
							items.add(item);
						}
					}
				} catch (Exception e) {
					LOGGER.debug(String.format("Citsmart Inventory > Erro ao recuperar informacao de item de storage do host de IP/name %s. Mensagem: %s.", ipHostName, (e.getMessage() != null ? e.getMessage() : "Nao informada")));
				}
			}
		} catch (final IOException e) {
			LOGGER.debug(String.format("Citsmart Inventory > Erro ao recuperar informacao de storage do host de IP/name %s. Mensagem: %s.", ipHostName, (e.getMessage() != null ? e.getMessage() : "Nao informada")));
		}
		return items;
	}

	private static Set<HostItem> getSoftwaresInfo(final String ipHostName, final int port, final String communityParm) {
		final Set<HostItem> items = new HashSet<>();
		try (final SNMPManager client = new SNMPManager("udp:" + ipHostName + "/" + port, communityParm)) {
			client.start();
			for (int i = 1; i < 500; i++) {
				try {
					final String sofName = client.getAsString(new OID(".1.3.6.1.2.1.25.6.3.1.2." + i));
					if (StringUtils.isNotBlank(sofName) && !sofName.trim().equalsIgnoreCase("Null")) {
						if (StringUtils.isNotBlank(sofName) && !sofName.trim().equalsIgnoreCase("noSuchInstance") && !sofName.trim().equalsIgnoreCase("noSuchObject")) {
							final HostItem item = new HostItem();
							item.setName(InventoryAttributes.SOFTWARES);
							final Map<String, String> detail = new HashMap<>();
							detail.put(InventoryAttributes.PUBLISHER, sofName);
							detail.put(InventoryAttributes.NAME, sofName);
							item.setProperties(detail);
							items.add(item);
						}
					}
				} catch (Exception e) {
					LOGGER.debug(String.format("Citsmart Inventory > Erro ao recuperar informacao de item de software do host de IP/name %s. Mensagem: %s.", ipHostName, (e.getMessage() != null ? e.getMessage() : "Nao informada")));
				}
			}
		} catch (final IOException e) {
			LOGGER.debug(String.format("Citsmart Inventory > Erro ao recuperar informacao de software do host de IP/name %s. Mensagem: %s.", ipHostName, (e.getMessage() != null ? e.getMessage() : "Nao informada")));
		}
		return items;
	}

	private static Set<HostItem> getNetworksInfo(final String ipHostName, final int port, final String communityParm) {
		final Set<HostItem> items = new HashSet<>();
		try (final SNMPManager client = new SNMPManager("udp:" + ipHostName + "/" + port, communityParm)) {
			client.start();
			for (int i = 1; i < 500; i++) {
				try {
					String descrNet = client.getAsString(new OID(".1.3.6.1.2.1.2.2.1.2." + i));
					if (StringUtils.isNotBlank(descrNet) && !descrNet.trim().equalsIgnoreCase("Null")) {
						descrNet = descrNet.replaceAll(":", "");
						descrNet = SNMPManager.fromHex(descrNet);
						String typeNet = client.getAsString(new OID(".1.3.6.1.2.1.2.2.1.3." + i));
						final String ipAddr = client.getAsString(new OID(".1.3.6.1.2.1.2.2.1.6." + i));
						final String speed = client.getAsString(new OID(".1.3.6.1.2.1.2.2.1.5." + i));
						String status = client.getAsString(new OID(".1.3.6.1.2.1.2.2.1.8." + i));

						if (StringUtils.isNotBlank(descrNet) && !descrNet.trim().equalsIgnoreCase("noSuchInstance") && !descrNet.trim().equalsIgnoreCase("noSuchObject")) {
							if (typeNet.trim().equalsIgnoreCase("6") || typeNet.trim().equalsIgnoreCase("71")) {
								if (typeNet.trim().equalsIgnoreCase("6")) {
									typeNet = "ethernetCsmacd";
								}
								if (typeNet.trim().equalsIgnoreCase("71")) {
									typeNet = "ieee80211";
								}
								if (status.trim().equalsIgnoreCase("1")) {
									status = "UP";
								}
								if (status.trim().equalsIgnoreCase("2")) {
									status = "DOWN";
								}
								if (status.trim().equalsIgnoreCase("6")) {
									status = "NOT PRESENT";
								}
								if (status.trim().equalsIgnoreCase("7")) {
									status = "lowerLayerDown";
								}
								if (status.trim().equalsIgnoreCase("5")) {
									status = "dormant";
								}
								final HostItem item = new HostItem();
								item.setName(InventoryAttributes.NETWORKS);
								final Map<String, String> detail = new HashMap<>();
								detail.put(InventoryAttributes.DESCRIPTION, descrNet);
								detail.put(InventoryAttributes.MACADDR, ipAddr);
								detail.put(InventoryAttributes.TYPE, typeNet);
								detail.put(InventoryAttributes.SPEED, speed);
								detail.put(InventoryAttributes.STATUS, status);
								item.setProperties(detail);
								items.add(item);
							}
						}
					}
				} catch (Exception e) {
					LOGGER.debug(String.format("Citsmart Inventory > Erro ao recuperar informacao de item de software do host de IP/name %s. Mensagem: %s.", ipHostName, (e.getMessage() != null ? e.getMessage() : "Nao informada")));
				}
			}
		} catch (final IOException e) {
			LOGGER.debug(String.format("Citsmart Inventory > Erro ao recuperar informacao de software do host de IP/name %s. Mensagem: %s.", ipHostName, (e.getMessage() != null ? e.getMessage() : "Nao informada")));
		}
		return items;
	}

	private static String fromHex(final String s) throws UnsupportedEncodingException {
		try {
			final byte bs[] = new byte[s.length() / 2];
			for (int i = 0; i < s.length(); i += 2) {
				bs[i / 2] = (byte) Integer.parseInt(s.substring(i, i + 2), 16);
			}
			return new String(bs, "UTF8");
		} catch (final Exception e) {
			return s;
		}
	}

	private static Long parseLong(final String toParse) {
		Long result = 0L;
		if (StringUtils.isNotBlank(toParse.trim())) {
			try {
				result = Long.parseLong(toParse.trim());
			} catch (final NumberFormatException e) {
				LOGGER.debug(String.format("Citsmart Inventory > Erro ao realizar parse de informacao de software. Mensagem: %s.", (e.getMessage() != null ? e.getMessage() : "Nao informada")));
			}
		}
		return result;
	}

	/**
	 * Start the Snmp session. If you forget the listen() method you will not
	 * get any answers because the communication is asynchronous and the
	 * listen() method listens for answers.
	 *
	 * @throws IOException
	 */
	private void start() throws IOException {
		final TransportMapping<?> transport = new DefaultUdpTransportMapping();
		snmp = new Snmp(transport);
		transport.listen();
	}

	@Override
	public void close() throws IOException {
		if (snmp != null) {
			snmp.close();
		}
	}

	/**
	 * Method which takes a single OID and returns the response from the agent
	 * as a String.
	 *
	 * @param oid
	 * @return
	 * @throws IOException
	 */
	private String getAsString(final OID oid) {
		String result = "";
		try {
			final ResponseEvent event = this.get(new OID[] { oid });
			if (event != null && event.getResponse() != null && event.getResponse().get(0) != null && event.getResponse().get(0).getVariable() != null) {
				result = event.getResponse().get(0).getVariable().toString();
			}
		} catch (final Exception e) {
			LOGGER.debug(String.format("Citsmart Inventory > Erro ao realizar parse de informacao de software. Mensagem: %s.", (e.getMessage() != null ? e.getMessage() : "Nao informada")));
			result = "";
		}
		return result.trim();
	}

	/**
	 * This method is capable of handling multiple OIDs
	 *
	 * @param oids
	 * @return
	 * @throws IOException
	 */
	private ResponseEvent get(final OID oids[]) throws IOException {
		final PDU pdu = new PDU();
		for (final OID oid : oids) {
			pdu.add(new VariableBinding(oid));
		}
		pdu.setType(PDU.GET);
		final ResponseEvent event = snmp.send(pdu, this.getTarget(), null);
		if (event != null) {
			return event;
		}
		throw new RuntimeException("GET timed out");
	}

	/**
	 * This method returns a Target, which contains information about where the
	 * data should be fetched and how.
	 *
	 * @return
	 */
	private Target getTarget() {
		final String snmpCommunity = (StringUtils.isBlank(community) && community != null ? community : "public");
		final Address targetAddress = GenericAddress.parse(address);
		final CommunityTarget target = new CommunityTarget();
		target.setCommunity(new OctetString(snmpCommunity));
		target.setAddress(targetAddress);
		target.setRetries(2);
		target.setTimeout(1500);
		target.setVersion(SnmpConstants.version2c);
		return target;
	}

}
