/**********************************LICENCA*GPLv2********************************************************************
* Copyright [2011,2012,2013,2014,2015,2016] da CentralIT Tecnologia da Informao Ltda (www.centralit.com.br)      *
*                                                                                                                  *
* Este arquivo  parte do programa/software: Citsmart (www.citsmart.com.br)                                        *
*                                                                                                                  *
* O Citsmart  um software livre; voc pode redistribui-lo e/ou modific-lo dentro dos termos da Licena           *
* Pblica Geral GNU como publicada pela Fundao do Software Livre (FSF); na verso 2 da Licena.                  *
*                                                                                                                  *
* Este programa/software  distribudo na esperana que possa ser til, mas SEM NENHUMA GARANTIA; sem uma          *
* garantia implcita de ADEQUAO a qualquer MERCADO ou APLICAO EM PARTICULAR. Veja a Licena Pblica Geral      *
* GNU/GPL em portugus para maiores detalhes.                                                                      *
*                                                                                                                  *
* Voc deve ter recebido uma cpia da Licena Pblica Geral GNU, sob o ttulo 'LICENCA.txt', junto com este        *
* programa/software, se no, acesse o Portal do Software Pblico Brasileiro no endereo www.softwarepublico.gov.br *
* ou escreva para a Fundao do Software Livre (FSF) Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301,USA  *
********************************************************************************************************************/
package br.com.centralit.batch;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import org.apache.log4j.Logger;

import br.com.centralit.agents.DotNetAgent;
import br.com.centralit.agents.IAgent;
import br.com.centralit.agents.SNMPAgent;
import br.com.centralit.bean.HostContent;
import br.com.centralit.bean.HostItem;
import br.com.centralit.bean.InventoryConfig;
import br.com.centralit.fila.IInventoryProducer;
import br.com.centralit.fila.InventoryProducer;
import br.com.centralit.pojo.CtInventario;
import br.com.centralit.util.Enumerados.StatusInventario;
import br.com.centralit.util.InventoryAttributes;
import br.com.centralit.util.JSONConverter;
import br.com.centralit.util.StatusRedeInventario;
import br.com.centralit.util.net.DiscoveryUtil;
import br.com.centralit.util.net.NetPing;

/**
 *
 * @author euler.ramos
 *
 */
public class Inventario implements Runnable {

	private static final Logger LOGGER = Logger.getLogger(Inventario.class);

	private final InventoryConfig config;
	private final String nomeHostIP;
	private IInventoryProducer inventoryProducer;

	public Inventario(final InventoryConfig config, final String nomeHostIP) {
		this.config = config;
		this.nomeHostIP = nomeHostIP;
	}

	@Override
	public void run() {
		HostContent resposta = new HostContent();

		LOGGER.debug("Citsmart Inventory > Iniciando inventário para host de IP/nome: " + nomeHostIP);

		if (this.executaPing(nomeHostIP)) {
			LOGGER.debug("Citsmart Inventory > O ping para o host " + nomeHostIP + " foi realizado com sucesso.");

			final ListIterator<IAgent> agents = this.getAgents().listIterator();

			do {
				final IAgent agent = agents.next();
				try {
					resposta = agent.doInventory(config, nomeHostIP);
				} catch (Exception e) {
					e.printStackTrace();
				}
			} while (resposta.getHostContent().isEmpty() && agents.hasNext());

			if (resposta.getHostContent().isEmpty()) {
				LOGGER.debug(String.format("Citsmart Inventory > Nao foi possivel inventariar o host de IP/nome: %s. Mensagem: Nenhum conteudo foi recuperado.", nomeHostIP));
				resposta = setUnreachable(StatusInventario.NAOINVENTARIADA);
			} else {
				if (this.config.getIgnorarIcsInventariados() != null && this.config.getIgnorarIcsInventariados().equalsIgnoreCase("S")) {
					DiscoveryUtil.inventariados.put(nomeHostIP, new Date());
				}

				for (HostItem hostItem : resposta.getHostContent()) {
					if (hostItem.getName().equalsIgnoreCase("HARDWARE")) {
						if (nomeHostIP.contains(".")) {
							hostItem.getProperties().put("IPCONF", nomeHostIP);
						}

						hostItem.getProperties().put("IDSTATUS", StatusInventario.INVENTARIADA.getId().toString());
						hostItem.getProperties().put("DTTIME", new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));

						if (this.config != null && this.config.getIdConexao() != null) {
							hostItem.getProperties().put("IDCON", this.config.getIdConexao().toString());
						}
					}
				}

				LOGGER.debug("Citsmart Inventory > Inventario para host de IP/nome " + nomeHostIP + " realizado.");
			}
		} else {
			LOGGER.debug(String.format("Citsmart Inventory > Nao foi possivel inventariar o host de IP/nome: %s. Mensagem: Nao respondeu ao ping.", nomeHostIP));
			resposta = setUnreachable(StatusInventario.INACESSIVEL);
		}

		this.sendToQueue(resposta);

		if (config != null && config.getConexaoEventMonitorUsuario() != null && config.getConexaoEventMonitorSenha() != null && config.getConexaoEventMonitorUrl() != null && !config.getConexaoEventMonitorUrl().isEmpty()) {
			this.enviarResultadoCitsmartEVM(resposta, config.getConexaoEventMonitorUsuario(), config.getConexaoEventMonitorSenha(), config.getConexaoEventMonitorUrl());
		}

		LOGGER.debug("Citsmart Inventory > Concluindo inventario para host de IP/nome: " + nomeHostIP);
	}

	private HostContent setUnreachable(StatusInventario statusInventario) {
		HostContent hostContent = new HostContent();

		final HostItem hardware = new HostItem();
		hardware.setName(InventoryAttributes.HARDWARE);
		final Map<String, String> hardwareDetail = new HashMap<>();
		if (nomeHostIP.contains(".")) {
			hardwareDetail.put("IPCONF", nomeHostIP);
			hardwareDetail.put(InventoryAttributes.IPADDR, nomeHostIP);
		} else {
			hardwareDetail.put(InventoryAttributes.NAME, nomeHostIP);
		}

		hardwareDetail.put("IDSTATUS", statusInventario.getId().toString());
		hardwareDetail.put("DTTIME", new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));

		if (this.config != null && this.config.getIdConexao() != null) {
			hardwareDetail.put("IDCON", this.config.getIdConexao().toString());
		}

		hardware.setProperties(hardwareDetail);
		hostContent.getHostContent().add(hardware);

		final HostItem network = new HostItem();
		network.setName(InventoryAttributes.NETWORKS);
		final Map<String, String> networkDetail = new HashMap<>();
		networkDetail.put(InventoryAttributes.STATUS, StatusRedeInventario.UNREACHABLE.name());

		network.setProperties(networkDetail);
		hostContent.getHostContent().add(network);

		return hostContent;
	}

	private boolean executaPing(final String hostIP) {
		return NetPing.isReachablebyPing(hostIP);
	}

	private List<IAgent> getAgents() {
		final List<IAgent> agents = new ArrayList<>();
		agents.add(new DotNetAgent());
		agents.add(new SNMPAgent());
		return agents;
	}

	private synchronized void sendToQueue(final HostContent resposta) {
		try {
			final CtInventario pojoInventario = new CtInventario();
			pojoInventario.setIp(nomeHostIP);
			pojoInventario.setInventario(resposta.getHostContent());

			getInventoryProducer().send(JSONConverter.toJson(pojoInventario));

			LOGGER.debug("Citsmart Inventory > Nova mensagem do host de IP/nome " + nomeHostIP + " enviada para a fila.");
		} catch (final Exception e) {
			LOGGER.debug("Citsmart Inventory > Nao foi possivel enviar mensagem do host de IP/nome " + nomeHostIP + " para a fila.", e);
		}
	}

	private synchronized void enviarResultadoCitsmartEVM(final HostContent resposta, final String conexaoEventMonitorUsuario, final String conexaoEventMonitorSenha, final String conexaoEventMonitorUrl) {
		try {
			LOGGER.debug("Citsmart Inventory > Enviando resultado ao servidor do Citsmart EVM.");

			String userPassword = conexaoEventMonitorUsuario + ":" + conexaoEventMonitorSenha;
			String encoding = new sun.misc.BASE64Encoder().encode(userPassword.getBytes());
			String urlCitsmartEventInventory = conexaoEventMonitorUrl;

			URL url = new URL(urlCitsmartEventInventory + "/ws/enviarEventoCTM");
			HttpURLConnection connection = (HttpURLConnection) url.openConnection();
			connection.setRequestProperty("Authorization", "Basic " + encoding);
			connection.setRequestMethod("PUT");
			connection.setRequestProperty("Content-Type", "application/json;charset=UTF-8");

			connection.setDoOutput(true);
			connection.setInstanceFollowRedirects(false);

			// Envia o objeto de conexão serializado em JSON
			final CtInventario pojoInventario = new CtInventario();
			pojoInventario.setIp(nomeHostIP);
			pojoInventario.setUserName("ADMIN");
			pojoInventario.setPassword("1");
			pojoInventario.setInventario(resposta.getHostContent());

			// Serializa e remove caracteres não imprimíveis
			final String dados = JSONConverter.toJson(pojoInventario).replaceAll("[\\x00\\x08\\x0B\\x0C\\x0E-\\x1F]", "");

			// Envia o inventário para o WS do EVM
			OutputStream os = connection.getOutputStream();
			os.write(dados.toString().getBytes("UTF-8"));
			os.flush();

			// Obtém a saída do webservice do Citsmart EVM
			String saidaDoServidor = "";
			BufferedReader br = new BufferedReader(new InputStreamReader(connection.getInputStream()));
			String output;

			while ((output = br.readLine()) != null) {
				saidaDoServidor += output;
			}

			// Imprime a resposta do servidor
			LOGGER.debug("Citsmart Inventory > Server response: " + saidaDoServidor);
			LOGGER.debug(String.format("Citsmart Inventory > ReponseCode: %s. ResponseMessage: %s", connection.getResponseCode(), connection.getResponseMessage()));

			connection.disconnect();
		} catch (IOException e) {
			LOGGER.debug(String.format("Citsmart Inventory > Nao foi possivel realizar a conexao com o Citsmart EVM. Mensagem: %s.", e.getMessage() != null ? e.getMessage() : "Nao informada"));
		}

	}

	public IInventoryProducer getInventoryProducer() {
		if (inventoryProducer == null) {
			return new InventoryProducer();
		}
		return inventoryProducer;
	}

}
