/**********************************LICENCA*GPLv2********************************************************************
* Copyright [2011,2012,2013,2014,2015,2016] da CentralIT Tecnologia da Informao Ltda (www.centralit.com.br)      *
*                                                                                                                  *
* Este arquivo  parte do programa/software: Citsmart (www.citsmart.com.br)                                        *
*                                                                                                                  *
* O Citsmart  um software livre; voc pode redistribui-lo e/ou modific-lo dentro dos termos da Licena           *
* Pblica Geral GNU como publicada pela Fundao do Software Livre (FSF); na verso 2 da Licena.                  *
*                                                                                                                  *
* Este programa/software  distribudo na esperana que possa ser til, mas SEM NENHUMA GARANTIA; sem uma          *
* garantia implcita de ADEQUAO a qualquer MERCADO ou APLICAO EM PARTICULAR. Veja a Licena Pblica Geral      *
* GNU/GPL em portugus para maiores detalhes.                                                                      *
*                                                                                                                  *
* Voc deve ter recebido uma cpia da Licena Pblica Geral GNU, sob o ttulo 'LICENCA.txt', junto com este        *
* programa/software, se no, acesse o Portal do Software Pblico Brasileiro no endereo www.softwarepublico.gov.br *
* ou escreva para a Fundao do Software Livre (FSF) Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301,USA  *
********************************************************************************************************************/
package br.com.centralit.batch;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

import br.com.centralit.bean.HostContent;
import br.com.centralit.bean.HostItem;
import br.com.centralit.bean.InventoryConfig;
import br.com.centralit.fila.IInventoryProducer;
import br.com.centralit.fila.InventoryProducer;
import br.com.centralit.pojo.CtInventario;
import br.com.centralit.util.Enumerados.StatusInventario;
import br.com.centralit.util.Enumerados.TipoInventario;
import br.com.centralit.util.Enumerados.TipoTempo;
import br.com.centralit.util.InventoryAttributes;
import br.com.centralit.util.JSONConverter;
import br.com.centralit.util.net.DiscoveryUtil;
import br.com.centralit.ws.GerenteConfiguracao;

/**
 * Thread que realiza a descoberta dos IC´s de uma faixa de rede passada como
 * parâmetro e gera o inventário destes IC´s ou de um único IC, quando foi
 * passado no parâmetro faixa o IP ou nome de um IC específico e o parâmetro
 * completo for falso.
 *
 * @author euler.ramos
 *
 */
public class ObtemInventario implements Runnable {

	private static final Logger LOGGER = Logger.getLogger(ObtemInventario.class);

	private final InventoryConfig config;

	private final DiscoveryUtil discovery = new DiscoveryUtil();
	private static final int numeroThreads = Runtime.getRuntime().availableProcessors() * 10;
	private final ExecutorService pool = Executors.newFixedThreadPool(numeroThreads);
	private InventoryProducer inventoryProducer;

	private TipoInventario tipoInventario = TipoInventario.COMPLETO;

	public ObtemInventario(final InventoryConfig config) {
		this.config = config;
	}

	public ObtemInventario(final InventoryConfig config, final TipoInventario tipoInventario) {
		this.config = config;
		this.tipoInventario = tipoInventario;
	}

	@Override
	public void run() {
		LOGGER.debug("Citsmart Inventory > Iniciando execucao do inventario (" + new SimpleDateFormat("dd/MM/yyyy HH:mm:ss.SSS").format(new Date()) + ").");

		if (config.getFaixa() != null && !config.getFaixa().isEmpty()) {
			List<NetMap> rede = new ArrayList<>();

			if (this.tipoInventario.equals(TipoInventario.COMPLETO)) {
				rede = discovery.obterEnderecosIP(config.getFaixa());
			} else {
				final NetMap netMap = new NetMap();
				netMap.setNomeHostIP(config.getFaixa());
				netMap.setNovo(true);

				rede.add(netMap);
			}

			for (final NetMap netMap : rede) {
				if (this.tipoInventario.equals(TipoInventario.DEMANDA)) {
					pool.execute(new Inventario(config, netMap.getNomeHostIP()));
				} else if ((this.config.getIgnorarIcsInventariados() != null && this.config.getIgnorarIcsInventariados().equalsIgnoreCase("S")) && (this.config.getTempoIgnorarIcs() != null && this.config.getTempoIgnorarIcs() > 0) && (this.config.getTipoIgnorarIcs() != null && this.config.getTipoIgnorarIcs() > 0)) {
					Date now = Calendar.getInstance().getTime();

					if (!DiscoveryUtil.inventariados.containsKey(netMap.getNomeHostIP()) || (Math.abs(now.getTime() - DiscoveryUtil.inventariados.get(netMap.getNomeHostIP()).getTime()) > TimeUnit.MILLISECONDS.convert(this.config.getTempoIgnorarIcs(), TipoTempo.fromId(this.config.getTipoIgnorarIcs()).getTimeUnit()))) {
						pool.execute(new Inventario(config, netMap.getNomeHostIP()));
					} else {
						HostContent hostContent = new HostContent();

						final HostItem hardware = new HostItem();
						hardware.setName(InventoryAttributes.HARDWARE);
						final Map<String, String> hardwareDetail = new HashMap<>();
						if (netMap.getNomeHostIP().contains(".")) {
							hardwareDetail.put("IPCONF", netMap.getNomeHostIP());
							hardwareDetail.put(InventoryAttributes.IPADDR, netMap.getNomeHostIP());
						} else {
							hardwareDetail.put(InventoryAttributes.NAME, netMap.getNomeHostIP());
						}

						hardwareDetail.put("IDSTATUS", StatusInventario.IGNORADA.getId().toString());
						hardwareDetail.put("DTTIME", new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));
						hardwareDetail.put("IDCON", this.config.getIdConexao().toString());

						hardware.setProperties(hardwareDetail);
						hostContent.getHostContent().add(hardware);

						sendToQueue(netMap.getNomeHostIP(), hostContent);
						LOGGER.debug(String.format("Citsmart Inventory > A opção de ignorar maquinas já inventariadas esta ativa. O host de IP/nome: %s ja foi inventariado.", netMap.getNomeHostIP()));
					}
				} else {
					pool.execute(new Inventario(config, netMap.getNomeHostIP()));
				}
			}

			try {
				pool.shutdown();

				if (pool.awaitTermination(5, TimeUnit.DAYS)) {
					LOGGER.debug(String.format("Citsmart Inventory > Concluida a execucao de inventario (" + new SimpleDateFormat("dd/MM/yyyy HH:mm:ss.SSS").format(new Date()) + "). Quantidade de hosts: %s.", rede.size()));

					if (tipoInventario.equals(TipoInventario.COMPLETO)) {
						GerenteConfiguracao.isInventarioExecutando = false;
					}
				} else {
					LOGGER.debug(String.format("Citsmart Inventory > Timeout na execucao de inventario (" + new SimpleDateFormat("dd/MM/yyyy HH:mm:ss.SSS").format(new Date()) + "). Quantidade de hosts: %s.", rede.size()));

					if (tipoInventario.equals(TipoInventario.COMPLETO)) {
						GerenteConfiguracao.isInventarioExecutando = false;
					}
				}

				pool.shutdown();
			} catch (final InterruptedException e) {
				LOGGER.debug("Citsmart Inventory > Erro na execucao de inventario (" + new SimpleDateFormat("dd/MM/yyyy HH:mm:ss.SSS").format(new Date()) + ").", e);
				GerenteConfiguracao.isInventarioExecutando = false;
			}
		}
	}

	private synchronized void sendToQueue(final String nomeHostIP, final HostContent resposta) {
		try {
			final CtInventario pojoInventario = new CtInventario();
			pojoInventario.setIp(nomeHostIP);
			pojoInventario.setInventario(resposta.getHostContent());

			getInventoryProducer().send(JSONConverter.toJson(pojoInventario));

			LOGGER.debug("Citsmart Inventory > Nova mensagem do host de IP/nome " + nomeHostIP + " enviada para a fila.");
		} catch (final Exception e) {
			LOGGER.debug("Citsmart Inventory > Nao foi possivel enviar mensagem do host de IP/nome " + nomeHostIP + " para a fila.", e);
		}
	}

	public IInventoryProducer getInventoryProducer() {
		if (inventoryProducer == null) {
			return new InventoryProducer();
		}
		return inventoryProducer;
	}
}
