/**********************************LICENCA*GPLv2********************************************************************
* Copyright [2011,2012,2013,2014,2015,2016] da CentralIT Tecnologia da Informao Ltda (www.centralit.com.br)      *
*                                                                                                                  *
* Este arquivo  parte do programa/software: Citsmart (www.citsmart.com.br)                                        *
*                                                                                                                  *
* O Citsmart  um software livre; voc pode redistribui-lo e/ou modific-lo dentro dos termos da Licena           *
* Pblica Geral GNU como publicada pela Fundao do Software Livre (FSF); na verso 2 da Licena.                  *
*                                                                                                                  *
* Este programa/software  distribudo na esperana que possa ser til, mas SEM NENHUMA GARANTIA; sem uma          *
* garantia implcita de ADEQUAO a qualquer MERCADO ou APLICAO EM PARTICULAR. Veja a Licena Pblica Geral      *
* GNU/GPL em portugus para maiores detalhes.                                                                      *
*                                                                                                                  *
* Voc deve ter recebido uma cpia da Licena Pblica Geral GNU, sob o ttulo 'LICENCA.txt', junto com este        *
* programa/software, se no, acesse o Portal do Software Pblico Brasileiro no endereo www.softwarepublico.gov.br *
* ou escreva para a Fundao do Software Livre (FSF) Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301,USA  *
********************************************************************************************************************/
package br.com.centralit.util.net;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import br.com.centralit.batch.NetMap;

/**
 * Classe utilitária responsável por gerar uma lista de IPs que serão inventariados
 *
 * @author euler.ramos
 *
 */
public class DiscoveryUtil {
	public static Map<String, Date> inventariados = new HashMap<String, Date>();

	/**
	 * Gera uma lista com todos os IPs constantes nas faixas de IP passadas no parâmetro faixaIP separadas por ponto e vírgula, ela
	 * pode receber a listaIPs já cadastradas no Citsmart para ir completando com os restantes existentes nas faixas
	 *
	 * <p>
	 * <strong>IMPORTANTE</strong>: Devido a urgência não foi criado um código para validação do que é passado como faixa de rede. Então é muito importante que a String faixaIP
	 * seja passada com valores válidos.
	 * </p>
	 *
	 * Vide: <a href="http://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless_Inter-Domain_Routing#CIDR_notation</a> <br>
	 * Usamos nos testes a CIDR : a.b.0.0/11 e foram gerados 2.097.152 endereços de IP! <br>
	 *
	 * <p>
	 * Exemplos de faixas válidas:<br>
	 * "10.2.1.0/24;200.100.5.1-200.100.5.3;192.168.0.1/30"<br>
	 * "10.2.1.1-10.2.1.255; 192.168.0.1/13; 200.100.5.1-200.100.5.3"
	 * </p>
	 *
	 * @param faixaIP
	 * @param listaIPs
	 * @return
	 *
	 */
	public List<NetMap> obterEnderecosIP(String faixaIP) {
		final List<NetMap> listaIPs = new ArrayList<NetMap>();

		NetMap netMap;

		if (faixaIP != null && !faixaIP.isEmpty()) {
			final String[] strFaixas = faixaIP.split(";");

			if (strFaixas != null && strFaixas.length > 0) {
				for (int i = 0; i < strFaixas.length; i++) {
					if (strFaixas[i] != null && !strFaixas[i].trim().isEmpty()) {
						final String[] ips = strFaixas[i].split("-");

						if (ips != null && ips.length > 0) {
							if (ips.length == 2 && (ips[0] == null || ips[0].trim().isEmpty()) && (ips[1] == null || ips[1].trim().isEmpty())) {
								continue;
							}

							if (ips.length == 1 || (ips.length == 2 && (ips[1] == null || ips[1].trim().isEmpty()))) {
								// O formato CIDR é 10.0.0.1/15 ou 192.168.1.255/24
								final Subnet subNet = new Subnet(ips[0].trim());
								ips[0] = subNet.getInfo().getLowAddress();
								ips[1] = subNet.getInfo().getHighAddress();
							}

							IPAddress ip1 = new IPAddress(ips[0]);
							IPAddress ip2 = new IPAddress(ips[1]);

							while (ip1.getValue() <= ip2.getValue()) {
								netMap = new NetMap();
								netMap.setNomeHostIP(ip1.toString());
								netMap.setNovo(true);

								if (listaIPs == null || !listaIPs.contains(netMap)) {
									listaIPs.add(netMap);
								}

								ip1 = ip1.next();
							}
						}
					}
				}
			}
		}

		if (listaIPs != null) {
			if (validaTodosInventariados(listaIPs)) {
				DiscoveryUtil.inventariados = new HashMap<String, Date>();
			}
		}

		return listaIPs;
	}

	public boolean validaTodosInventariados(List<NetMap> listaIPs) {
		Date now = Calendar.getInstance().getTime();

		if (listaIPs != null) {
			for (NetMap nMap : listaIPs) {
				if (!inventariados.containsKey(nMap.getNomeHostIP())
						|| Math.abs(now.getTime() - DiscoveryUtil.inventariados.get(nMap.getNomeHostIP()).getTime()) > TimeUnit.MILLISECONDS.convert(1, TimeUnit.DAYS)) {
					return false;
				}
			}
		}

		return true;
	}

}
