/**********************************LICENCA*GPLv2********************************************************************
* Copyright [2011,2012,2013,2014,2015,2016] da CentralIT Tecnologia da Informao Ltda (www.centralit.com.br)      *
*                                                                                                                  *
* Este arquivo  parte do programa/software: Citsmart (www.citsmart.com.br)                                        *
*                                                                                                                  *
* O Citsmart  um software livre; voc pode redistribui-lo e/ou modific-lo dentro dos termos da Licena           *
* Pblica Geral GNU como publicada pela Fundao do Software Livre (FSF); na verso 2 da Licena.                  *
*                                                                                                                  *
* Este programa/software  distribudo na esperana que possa ser til, mas SEM NENHUMA GARANTIA; sem uma          *
* garantia implcita de ADEQUAO a qualquer MERCADO ou APLICAO EM PARTICULAR. Veja a Licena Pblica Geral      *
* GNU/GPL em portugus para maiores detalhes.                                                                      *
*                                                                                                                  *
* Voc deve ter recebido uma cpia da Licena Pblica Geral GNU, sob o ttulo 'LICENCA.txt', junto com este        *
* programa/software, se no, acesse o Portal do Software Pblico Brasileiro no endereo www.softwarepublico.gov.br *
* ou escreva para a Fundao do Software Livre (FSF) Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301,USA  *
********************************************************************************************************************/
package br.com.centralit.util.net;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Subnet {

    private static final String IP_ADDRESS = "(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})";
    private static final String SLASH_FORMAT = Subnet.IP_ADDRESS + "/(\\d{1,3})";
    private static final Pattern addressPattern = Pattern.compile(Subnet.IP_ADDRESS);
    private static final Pattern cidrPattern = Pattern.compile(Subnet.SLASH_FORMAT);
    private static final int NBITS = 32;

    private int netmask = 0;
    private int address = 0;
    private int network = 0;
    private int broadcast = 0;

    public Subnet() {}

    /**
     * Constructor that takes a CIDR-notation string, e.g. "192.168.0.1/16"
     *
     * @param cidrNotation
     *            A CIDR-notation string, e.g. "192.168.0.1/16"
     */
    public Subnet(final String cidrNotation) {
        this.calculate(cidrNotation);
    }

    /**
     * Constructor that takes two dotted decimal addresses.
     *
     * @param address
     *            An IP address, e.g. "192.168.0.1"
     * @param mask
     *            A dotted decimal netmask e.g. "255.255.0.0"
     */
    public Subnet(final String address, final String mask) {
        this.calculate(this.toCidrNotation(address, mask));
    }

    /**
     * Convenience container for subnet summary information.
     *
     */
    public final class SubnetInfo {

        private SubnetInfo() {}

        private int netmask() {
            return netmask;
        }

        private int network() {
            return network;
        }

        private int address() {
            return address;
        }

        private int broadcast() {
            return broadcast;
        }

        private int low() {
            return this.network() + 1;
        }

        private int high() {
            return this.broadcast() - 1;
        }

        public boolean isInRange(final String address) {
            return this.isInRange(Subnet.this.toInteger(address));
        }

        private boolean isInRange(final int address) {
            return address - this.low() <= this.high() - this.low();
        }

        public String getBroadcastAddress() {
            return Subnet.this.format(Subnet.this.toArray(this.broadcast()));
        }

        public String getNetworkAddress() {
            return Subnet.this.format(Subnet.this.toArray(this.network()));
        }

        public String getNetmask() {
            return Subnet.this.format(Subnet.this.toArray(this.netmask()));
        }

        public String getAddress() {
            return Subnet.this.format(Subnet.this.toArray(this.address()));
        }

        public String getLowAddress() {
            return Subnet.this.format(Subnet.this.toArray(this.low()));
        }

        public String getHighAddress() {
            return Subnet.this.format(Subnet.this.toArray(this.high()));
        }

        public int getAddressCount() {
            return this.broadcast() - this.low();
        }

        public int asInteger(final String address) {
            return Subnet.this.toInteger(address);
        }

        public String getCidrSignature() {
            return Subnet.this.toCidrNotation(Subnet.this.format(Subnet.this.toArray(this.address())), Subnet.this.format(Subnet.this.toArray(this.netmask())));
        }

        public String[] getAllAddresses() {
            final String[] addresses = new String[this.getAddressCount()];
            for (int add = this.low(), j = 0; add <= this.high(); ++add, ++j) {
                addresses[j] = Subnet.this.format(Subnet.this.toArray(add));
            }
            return addresses;
        }

    }

    /**
     * Return a {@link SubnetInfo} instance that contains subnet-specific statistics
     *
     * @return
     */
    public final SubnetInfo getInfo() {
        return new SubnetInfo();
    }

    /*
     * Initialize the internal fields from the supplied CIDR mask
     */
    private void calculate(final String mask) {
        final Matcher matcher = Subnet.cidrPattern.matcher(mask);

        if (matcher.matches()) {
            address = this.matchAddress(matcher);

            /* Create a binary netmask from the number of bits specification /x */
            final int cidrPart = this.rangeCheck(Integer.parseInt(matcher.group(5)), 0, Subnet.NBITS - 1);
            for (int j = 0; j < cidrPart; ++j) {
                netmask |= 1 << 31 - j;
            }

            /* Calculate base network address */
            network = address & netmask;

            /* Calculate broadcast address */
            broadcast = network | ~netmask;
        } else {
            throw new IllegalArgumentException("Could not parse [" + mask + "]");
        }
    }

    /*
     * Convert a dotted decimal format address to a packed integer format
     */
    private int toInteger(final String address) {
        final Matcher matcher = Subnet.addressPattern.matcher(address);
        if (matcher.matches()) {
            return this.matchAddress(matcher);
        } else {
            throw new IllegalArgumentException("Could not parse [" + address + "]");
        }
    }

    /*
     * Convenience method to extract the components of a dotted decimal address and
     * pack into an integer using a regex match
     */
    private int matchAddress(final Matcher matcher) {
        int addr = 0;
        for (int i = 1; i <= 4; ++i) {
            final int n = this.rangeCheck(Integer.parseInt(matcher.group(i)), 0, 255);
            addr |= (n & 0xff) << 8 * (4 - i);
        }
        return addr;
    }

    /*
     * Convert a packed integer address into a 4-element array
     */
    private int[] toArray(final int val) {
        final int ret[] = new int[4];
        for (int j = 3; j >= 0; --j) {
            ret[j] |= val >>> 8 * (3 - j) & 0xff;
        }
        return ret;
    }

    /*
     * Convert a 4-element array into dotted decimal format
     */
    private String format(final int[] octets) {
        final StringBuilder str = new StringBuilder();
        for (int i = 0; i < octets.length; ++i) {
            str.append(octets[i]);
            if (i != octets.length - 1) {
                str.append(".");
            }
        }
        return str.toString();
    }

    /*
     * Convenience function to check integer boundaries
     */
    private int rangeCheck(final int value, final int begin, final int end) {
        if (value >= begin && value <= end) {
            return value;
        }

        throw new IllegalArgumentException("Value out of range: [" + value + "]");
    }

    /*
     * Count the number of 1-bits in a 32-bit integer using a divide-and-conquer strategy
     * see Hacker's Delight section 5.1
     */
    int pop(int x) {
        x = x - (x >>> 1 & 0x55555555);
        x = (x & 0x33333333) + (x >>> 2 & 0x33333333);
        x = x + (x >>> 4) & 0x0F0F0F0F;
        x = x + (x >>> 8);
        x = x + (x >>> 16);
        return x & 0x0000003F;
    }

    /*
     * Convert two dotted decimal addresses to a single xxx.xxx.xxx.xxx/yy format
     * by counting the 1-bit population in the mask address. (It may be better to count
     * NBITS-#trailing zeroes for this case)
     */
    private String toCidrNotation(final String addr, final String mask) {
        return addr + "/" + this.pop(this.toInteger(mask));
    }

}
