/**********************************LICENCA*GPLv2********************************************************************
* Copyright [2011,2012,2013,2014,2015,2016] da CentralIT Tecnologia da Informao Ltda (www.centralit.com.br)      *
*                                                                                                                  *
* Este arquivo  parte do programa/software: Citsmart (www.citsmart.com.br)                                        *
*                                                                                                                  *
* O Citsmart  um software livre; voc pode redistribui-lo e/ou modific-lo dentro dos termos da Licena           *
* Pblica Geral GNU como publicada pela Fundao do Software Livre (FSF); na verso 2 da Licena.                  *
*                                                                                                                  *
* Este programa/software  distribudo na esperana que possa ser til, mas SEM NENHUMA GARANTIA; sem uma          *
* garantia implcita de ADEQUAO a qualquer MERCADO ou APLICAO EM PARTICULAR. Veja a Licena Pblica Geral      *
* GNU/GPL em portugus para maiores detalhes.                                                                      *
*                                                                                                                  *
* Voc deve ter recebido uma cpia da Licena Pblica Geral GNU, sob o ttulo 'LICENCA.txt', junto com este        *
* programa/software, se no, acesse o Portal do Software Pblico Brasileiro no endereo www.softwarepublico.gov.br *
* ou escreva para a Fundao do Software Livre (FSF) Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301,USA  *
********************************************************************************************************************/
package br.com.centralit.ws;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.ejb.ConcurrencyManagement;
import javax.ejb.ConcurrencyManagementType;
import javax.ejb.ScheduleExpression;
import javax.ejb.Singleton;
import javax.ejb.Startup;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerConfig;
import javax.ejb.TimerService;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;

import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.quartz.CronExpression;

import br.com.centralit.batch.ObtemInventario;
import br.com.centralit.bean.InventoryConfig;
import br.com.centralit.util.Enumerados.TipoInventario;
import br.com.centralit.util.JSONConverter;
import br.com.centralit.util.LogicException;


@Startup
@Singleton
@TransactionManagement(value = TransactionManagementType.BEAN)
@ConcurrencyManagement(value = ConcurrencyManagementType.BEAN)
public class GerenteConfiguracao implements IGerenteTarefas {

	private static final Logger LOGGER = Logger.getLogger(GerenteConfiguracao.class);

	@Resource
	private TimerService timerService;

	/**
	 * Quando a sincronização dos agendamentos com o WS do Citsmart falha, esta
	 * variavel deve ser setada para true
	 */
	private boolean reagendarSincronizacao = false;
	
	/**
	 * Indica se o inventário está em execução.
	 */
	public static boolean isInventarioExecutando;

	@Override
	@PostConstruct
	public void inicializarBean() {
		this.sincronizarHorariosInventarioComWSCitsmart();
	}

	@Override
	public void sincronizarHorariosInventarioComWSCitsmart() {
		String dados = new JSONObject().put("userName", System.getProperty("citsmart.login")).put("password", System.getProperty("citsmart.password")).put("idCitsmartInventory", System.getProperty("citsmart.inventory.id")).toString();

		try {
			if (System.getProperty("citsmart.host") != null && !System.getProperty("citsmart.host").isEmpty() && System.getProperty("citsmart.port") != null && !System.getProperty("citsmart.port").isEmpty() && System.getProperty("citsmart.context") != null && !System.getProperty("citsmart.context").isEmpty()) {
				LOGGER.info("Citsmart Inventory > Importando agendamento do WebService Citsmart.");

				final URL url = new URL(String.format("http://%s:%s/%s/citsmartInventory/obtemParametrosInicializacao", System.getProperty("citsmart.host"), System.getProperty("citsmart.port"), System.getProperty("citsmart.context")));
				final HttpURLConnection connection = (HttpURLConnection) url.openConnection();

				connection.setRequestMethod("PUT");
				connection.setRequestProperty("Content-Type", "application/json");
				connection.setDoOutput(true);
				connection.setInstanceFollowRedirects(false);

				try (final OutputStream os = connection.getOutputStream()) {
					os.write(dados.toString().getBytes());
					os.flush();
				}

				// Obtém a saída do webservice do Citsmart
				final StringBuilder saidaDoServidor = new StringBuilder();
				try (final InputStreamReader is = new InputStreamReader(connection.getInputStream()); final BufferedReader br = new BufferedReader(is)) {
					String output;
					while ((output = br.readLine()) != null) {
						saidaDoServidor.append(output);
					}
				}

				if (connection.getResponseCode() == 200) {
					LOGGER.debug("Citsmart Inventory > Conexao com o Citsmart realizada.");
				}

				LOGGER.debug(String.format("Citsmart Inventory > ReponseCode: %s. ResponseMessage: %s", connection.getResponseCode(), connection.getResponseMessage()));

				if (saidaDoServidor != null && saidaDoServidor.length() > 0) {
					LOGGER.debug(String.format("Citsmart Inventory > Resultado da importacao: %s.", saidaDoServidor.toString()));

					InventoryConfig inventarioDTOTmp = JSONConverter.fromJson(saidaDoServidor.toString(), InventoryConfig.class);

					if (inventarioDTOTmp != null && inventarioDTOTmp.getExpressaoCron() != null) {
						this.processCronExpression(inventarioDTOTmp.getExpressaoCron(), inventarioDTOTmp);

						InventoryWS.cronDoInventario = inventarioDTOTmp.getExpressaoCron();
					}
				} else {
					LOGGER.debug("Citsmart Inventory > Resultado da importacao: Nenhum agendamento importado.");
				}

				if (reagendarSincronizacao == true) {
					this.cancelarReagendamento();
					reagendarSincronizacao = false;
				}

				connection.disconnect();
			} else {
				throw new LogicException("Host e/ou porta para o Citsmart nao foi informado. Verificar arquivo inventory_config.properties.");
			}
		} catch (final Exception e) {
			LOGGER.debug(String.format("Citsmart Inventory > Nao foi possivel realizar a conexao com o Citsmart. Mensagem: %s.", (e.getMessage() != null ? e.getMessage() : "Nao informada")));

			if (reagendarSincronizacao == false) {
				this.reagendarSincronizacao = true;
				final InventoryConfig inventarioDTOTmp = new InventoryConfig();
				inventarioDTOTmp.setConexaoFalhou(true);
				this.reagendarSincronizarHorariosInventarioWSCitsmart(inventarioDTOTmp);
			}
		}
	}

	public void realizarInventario(final InventoryConfig config) {
		if (config != null) {
			GerenteConfiguracao.isInventarioExecutando = true;
			
			final ObtemInventario obtemInventario = new ObtemInventario(config, TipoInventario.COMPLETO);
			obtemInventario.run();
		}
	}

	/*
	 * Este método é chamado toda vez que chega o momento de disparar alguma
	 * tarefa
	 */
	@Override
	@Timeout
	public void metodoTimeout(final Timer timer) {
		if (!GerenteConfiguracao.isInventarioExecutando) {
			final InventoryConfig inventoryConfig = (InventoryConfig) timer.getInfo();
			try {
				if (inventoryConfig.isConexaoFalhou()) {
					this.sincronizarHorariosInventarioComWSCitsmart();
				} else {
					realizarInventario(inventoryConfig);
				}
			} catch (final Exception e) {
				LOGGER.debug(e.getMessage(), e);
			}
		} else {
			LOGGER.debug("Citsmart Inventory > Tentativa de execucao do inventario cancelada. Ja existe um inventario em execucao.");
		}
	}

	/**
	 * Quando a sincronização dos horários de para realização dos inventários do
	 * Citsmart Inventory com o Citsmart falha, e necessário reagendar uma nova
	 * sincronização com o Citsmart
	 */
	private void reagendarSincronizarHorariosInventarioWSCitsmart(final InventoryConfig inventoryConfig) {
		// Reagendando uma nova tentativa de sincronização com o Citsmart daqui
		// a 1 minuto
		final String cron = "0 */1 * * * ? *";

		this.processCronExpression(cron, inventoryConfig);

		LOGGER.warn("Citsmart Inventory > Uma nova tentativa de sincronização dos agendamentos com o WS do Citsmart sera executada em 1 minuto.");
	}

	private void processCronExpression(final String cron, final InventoryConfig inventoryConfig) {
		if (CronExpression.isValidExpression(cron)) {
			final String[] vetorCron = cron.split(" ");
			final ScheduleExpression schedule = new ScheduleExpression();

			for (int j = 0; j < vetorCron.length; j++) {
				// Ignora a expressão ? (interrogação)
				if (!vetorCron[j].equals("?")) {
					// Segundo
					if (j == 0) {
						schedule.second(vetorCron[j]);
					}

					// Minuto
					if (j == 1) {
						schedule.minute(vetorCron[j]);
					}

					// Hora
					if (j == 2) {
						schedule.hour(vetorCron[j]);
					}

					// Dia do Mes
					if (j == 3) {
						schedule.dayOfMonth(vetorCron[j]);
					}

					// Mes
					if (j == 4) {
						schedule.month(vetorCron[j]);
					}

					// Dia da Semana
					if (j == 5) {
						schedule.dayOfWeek(vetorCron[j]);
					}

					// Ano
					if (j == 6) {
						schedule.year(vetorCron[j]);
					}
				}
			}

			final TimerConfig timerConfig = new TimerConfig(inventoryConfig, false);

			try {
				timerService.createCalendarTimer(schedule, timerConfig);

				LOGGER.debug("Citsmart Inventory > Processou a expressão CRON e criou-se um novo agendador de inventário. Expressão: " + inventoryConfig.getExpressaoCron());
			} catch (Exception e) {
				LOGGER.debug("Citsmart Inventory > Exception ao tentar criar novo timer. Mensagem: " + e.getMessage());
			}
		} else {
			LOGGER.debug("Citsmart Inventory > Nao foi possivel processar expressao cron. Mensagem: Expressao cron invalida.");
		}
	}

	/**
	 * Paraliza um timer específico
	 *
	 * @param timerService
	 * @param evmCronDTO
	 */
	@Override
	public void cancelarTimer(final InventoryConfig inventoryConfig) { 
		for (final Object obj : timerService.getTimers()) {
			final Timer t = (Timer) obj;

			final InventoryConfig cdt = (InventoryConfig) t.getInfo();
			if (cdt.equals(inventoryConfig)) {
				t.cancel();
				LOGGER.debug("Citsmart Inventory > Cancelou o tempororizador com expressao cron \"" + String.valueOf(cdt.getExpressaoCron()) + "\" !!");
			}
		}
	}

	/**
	 * Cancela todos os agendamentos (timers)
	 *
	 * @param timerService
	 * @param evmCronDTO
	 */
	@Override
	public void cancelarTimers() {
		for (final Object obj : timerService.getTimers()) {
			final Timer t = (Timer) obj;

			t.cancel();
		}
		LOGGER.debug("Citsmart Inventory > Cancelou os temporizadores/agendamentos anteriores.");
	}

	/**
	 * Cancela o reagendamento de sincronização com o Citsmart
	 *
	 */
	public void cancelarReagendamento() {
		for (final Object obj : timerService.getTimers()) {
			final Timer t = (Timer) obj;

			final InventoryConfig cdt = (InventoryConfig) t.getInfo();
			if (cdt.isConexaoFalhou() == true) {
				t.cancel();
				LOGGER.debug("Citsmart Inventory > Cancelou o reagendamento de sincronização com o Citsmart.");
			}
		}
	}

	@Override
	public void atualizarTemporizador(final InventoryConfig inventoryConfig) {
		this.cancelarTimer(inventoryConfig);

		LOGGER.debug("Citsmart Inventory > Atualizando agendamento do WebService Citsmart.");

		this.processCronExpression(inventoryConfig.getExpressaoCron(), inventoryConfig);

		LOGGER.debug("Citsmart Inventory > Atualizou o tempororizador com a expressão \"" + inventoryConfig.getExpressaoCron() + "\" com os dados do webservice Citsmart.");
	}

}
