<?php
/********************************************************************************
*  Copyright 2015 Conab - Companhia Nacional de Abastecimento                   *
*                                                                               *
*  Este arquivo  parte do Sistema SIAUDI.                                      *
*                                                                               *
*  SIAUDI   um software livre; voc pode redistribui-lo e/ou                   *
*  modific-lo sob os termos da Licena Pblica Geral GNU conforme              *
*  publicada pela Free Software Foundation; tanto a verso 2 da                 *
*  Licena, como (a seu critrio) qualquer verso posterior.                    *
*                                                                               *
*  SIAUDI  distribudo na expectativa de que seja til,                        *
*  porm, SEM NENHUMA GARANTIA; nem mesmo a garantia implcita                  *
*  de COMERCIABILIDADE OU ADEQUAO A UMA FINALIDADE ESPECFICA.                *
*  Consulte a Licena Pblica Geral do GNU para mais detalhes em portugus:     *
*  http://creativecommons.org/licenses/GPL/2.0/legalcode.pt                     *
*                                                                               *
*  Voc deve ter recebido uma cpia da Licena Pblica Geral do GNU             *
*  junto com este programa; se no, escreva para a Free Software                *
*  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    *
*                                                                               *
*  Sistema   : SIAUDI - Sistema de Auditoria Interna                            *
*  Data      : 05/2015                                                          *
*                                                                               *
********************************************************************************/
?>
<?php
/**
 * Classe para manipulao/transformao de strings especficas no oferecidas 
 * via funes nativas do PHP.
 * 
 * Todos os mtodos devem transformar a string de alguma forma.
 * 
 * Esta classe teve origem de metodos retirados da antiga CMask, que passou a
 * acomodar helpers de validao e tratamento de datas, dentre outros, alm da 
 * falta de padronizao nos nomes dos mtodos.
 * 
 * As manipulaes de Data no foram migradas, para dar preferencia ao uso da 
 * classe DateTime nativa do PHP.
 * 
 * Ao criar um metodo, favor siga este processo:
 * 1. Verifique se o mtodo que precisa j no existe, para no haver duplicao
 * 2. Mantenha o mesmo padro de nomenclatura
 * 3. Verifique novamente se o mtodo que precisa j no existe. :)
 * 4. Evite ao mximo o acoplamento.
 * 
 * @package \common\util
 */
class CTransform
{
    /**
     * Remove os acentos e caracteres especiais e transformar para caixa alta
     * 
     * @param string $string String a ser manipulada
     * @return string String em maiusculo e sem caracteres especiais
     */
    public static function removeEspeciais($string)
    {
        return preg_replace('/[`^~\'"]/', null, iconv('ISO-8859-1', 'ASCII//TRANSLIT', $string));
    }


    /**
     * Transforma uma string numrica em valor (float), removendo os separadores
     * de milhar.
     * 
     * Exemplo: CTransform::stringParaValor('9.999,99'); // retorna (float) 9999.99
     * 
     * @param string $string A ser transformada
     * @return float|integer|null Valor transformado ou nulo caso no seja uma string numrica
     */
    public static function stringParaValor($string)
    {
        if (empty($string)) {
            return null;
        }

        $string = str_replace(".", "", $string);
        $string = str_replace(",", ".", $string);
        if (strpos($string, '.')) {
            return (float) $string;
        }
        return (integer) $string;
    }

    /**
     * Trasnforma um numero em extenso, pt-BR, a partir do nmero.
     * 
     * Exemplo:
     *   CTransform::numeroParaExtenso(12345678.90, "real", "reais", "centavo", "centavos")
     * 
     * @author Chrystian Toigo
     * @since 06/12/2011
     * @param float $valor Nmero (ponto flutuante) a ser convertido
     * @return string|null Nmero por extenso, ou nulo caso um nmero no seja passado
     */
    public static function numeroParaExtenso($valor, $moedaSing, $moedaPlur, $centSing, $centPlur)
    {
        if (!is_numeric($valor)) {
            return null;
        }

        $centenas = array(0,
            array(0, "cento", "cem"),
            array(0, "duzentos", "duzentos"),
            array(0, "trezentos", "trezentos"),
            array(0, "quatrocentos", "quatrocentos"),
            array(0, "quinhentos", "quinhentos"),
            array(0, "seiscentos", "seiscentos"),
            array(0, "setecentos", "setecentos"),
            array(0, "oitocentos", "oitocentos"),
            array(0, "novecentos", "novecentos"));

        $dezenas = array(0,
            "dez",
            "vinte",
            "trinta",
            "quarenta",
            "cinquenta",
            "sessenta",
            "setenta",
            "oitenta",
            "noventa");

        $unidades = array(0,
            "um",
            "dois",
            "trs",
            "quatro",
            "cinco",
            "seis",
            "sete",
            "oito",
            "nove");

        $excecoes = array(0,
            "onze",
            "doze",
            "treze",
            "quatorze",
            "quinze",
            "dezeseis",
            "dezesete",
            "dezoito",
            "dezenove");

        $extensoes = array(0,
            array(0, "", ""),
            array(0, "mil", "mil"),
            array(0, "milho", "milhes"),
            array(0, "bilho", "bilhes"),
            array(0, "trilho", "trilhes"));

        $valorForm = trim(number_format($valor, 2, ".", ","));

        $inicio = 0;

        if ($valor <= 0) {
            return ( $valorExt );
        }

        for ($conta = 0; $conta <= strlen($valorForm) - 1; $conta++) {
            if (strstr(",.", substr($valorForm, $conta, 1))) {
                $partes[] = str_pad(substr($valorForm, $inicio, $conta - $inicio), 3, " ", STR_PAD_LEFT);
                if (substr($valorForm, $conta, 1) == ".") {
                    break;
                }
                $inicio = $conta + 1;
            }
        }

        $centavos = substr($valorForm, strlen($valorForm) - 2, 2);

        if (!( count($partes) == 1 and intval($partes[0]) == 0 )) {
            for ($conta = 0; $conta <= count($partes) - 1; $conta++) {

                $centena = intval(substr($partes[$conta], 0, 1));
                $dezena = intval(substr($partes[$conta], 1, 1));
                $unidade = intval(substr($partes[$conta], 2, 1));

                if ($centena > 0) {

                    $valorExt .= $centenas[$centena][($dezena + $unidade > 0 ? 1 : 2)] . ( $dezena + $unidade > 0 ? " e " : "" );
                }

                if ($dezena > 0) {
                    if ($dezena > 1) {
                        $valorExt .= $dezenas[$dezena] . ( $unidade > 0 ? " e " : "" );
                    } elseif ($dezena == 1 and $unidade == 0) {
                        $valorExt .= $dezenas[$dezena];
                    } else {
                        $valorExt .= $excecoes[$unidade];
                    }
                }

                if ($unidade > 0 and $dezena != 1) {
                    $valorExt .= $unidades[$unidade];
                }

                if (intval($partes[$conta]) > 0) {
                    $valorExt .= " " . $extensoes[(count($partes) - 1) - $conta + 1][(intval($partes[$conta]) > 1 ? 2 : 1)];
                }

                if ((count($partes) - 1) > $conta and intval($partes[$conta]) > 0) {
                    $conta3 = 0;
                    for ($conta2 = $conta + 1; $conta2 <= count($partes) - 1; $conta2++) {
                        $conta3 += (intval($partes[$conta2]) > 0 ? 1 : 0);
                    }

                    if ($conta3 == 1 and intval($centavos) == 0) {
                        $valorExt .= " e ";
                    } elseif ($conta3 >= 1) {
                        $valorExt .= ", ";
                    }
                }
            }

            if (count($partes) == 1 and intval($partes[0]) == 1) {
                $valorExt .= $moedaSing;
            } elseif (count($partes) >= 3 and ((intval($partes[count($partes) - 1]) + intval($partes[count($partes) - 2])) == 0)) {
                $valorExt .= " de " + $moedaPlur;
            } else {
                $valorExt = trim($valorExt) . " " . $moedaPlur;
            }
        }

        if (intval($centavos) > 0) {

            $valorExt .= (!empty($valorExt) ? " e " : "");

            $dezena = intval(substr($centavos, 0, 1));
            $unidade = intval(substr($centavos, 1, 1));

            if ($dezena > 0) {
                if ($dezena > 1) {
                    $valorExt .= $dezenas[$dezena] . ( $unidade > 0 ? " e " : "" );
                } elseif ($dezena == 1 and $unidade == 0) {
                    $valorExt .= $dezenas[$dezena];
                } else {
                    $valorExt .= $excecoes[$unidade];
                }
            }

            if ($unidade > 0 and $dezena != 1) {
                $valorExt .= $unidades[$unidade];
            }

            $valorExt .= " " . ( intval($centavos) > 1 ? $centPlur : $centSing );
        }

        return ( $valorExt );
    }

    /**
     * Transforma o ms em extenso, pt-BR, a partir do nmero do ms.
     * 
     * Exemplo: CTransform::mesParaExtenso(2); // retorna "FEVEREIRO"
     * 
     * @param integer $mes Ms no formato numrico. ex.: 1, 2, 10, ...
     * @return string|null Ms por extenso ou nulo caso um mes invalido seja passado
     */
    public static function mesParaExtenso($mes)
    {
        if ($mes > 0 && $mes < 13) {
            $meses = array(
                "JANEIRO", "FEVEREIRO", "MARO", "ABRIL", "MAIO", "JUNHO",
                "JULHO", "AGOSTO", "SETEMBRO", "OUTUBRO", "NOVEMBRO", "DEZEMBRO"
            );
            return $meses[--$mes];
        }
        return null;
    }

    /**
     * Retorna uma enumerao (do portugus) com os tens de um determinado array.
     * 
     * Exemplo:
     *   entrada: CTransform::arrayParaEnumeracao(array("Arroz", "Feijo", "Ovo"))
     *   retorna: (string) "Arroz, Feijo e Ovo"
     * 
     * @author igor.costa
     * @param array $array O array desejado para transformao
     * @return string Com a enumerao em portugus.
     */
    public function arrayParaEnumeracao($array)
    {
        return join(' e ', array_filter(array_merge(array(join(', ', array_slice($array, 0, -1))), array_slice($array, -1))));
    }
}
